#include "pch.h"

#include "Mem.hpp"
#include "Common.hpp"
#include "CfgParse.hpp"
#include "Misc.hpp"
#include "Links.h"
#include "Areas.h"
#include "Msg.hpp"
#include "Tic.hpp"
#include "crc32.hpp"
#include "Temp.hpp"
#include "File.h"

#include "CfgLexems.hpp"
#include "WELexems.h"

#include "MsgAttrs.h"

#include "LineFormat.h"

#include "exceptions.h"
#include "singletons.h"

#include "actions/HelpAction.h"
#include "actions/ChangeGroupAction.h"

#include "AreasManager.h"
#include "LinksManager.h"
#include "Adrenalin.h"

// main class instance

CAdrenalin*	pAdrenalin;
// static members
TCHAR CAdrenalin::m_szStartupPath[MAX_PATH] = TEXT("");
TCHAR CAdrenalin::m_szStartupDrive[3] = TEXT("");		// <letter>':''\0'

class CSingletonDestroyer {
public:
	CSingletonDestroyer() {}
	~CSingletonDestroyer() {
		CDestructionManager::getInstance()->destroyAll();
	}
};

int _tmain(int argc, TCHAR **argv) {
	// on exit from main destructor of the underlying class will be called
	CSingletonDestroyer	singletonDestroyer;

	int	result = 0;

	try {

		pAdrenalin = new CAdrenalin( argc, argv );

		result = pAdrenalin->Execute();

		delete	pAdrenalin;
		

	} catch (CRuntimeException& e) {
		cerr << "\n[!] RuntimeException occured:\n" << e.getMessage().c_str() << endl;
		result = 1;
	}

	return result;
}


CAdrenalin::CAdrenalin(int argc, TCHAR **argv):
	tossingBadTics(false)
{
    // keep start time
	::GetLocalTime( &startTime );

	// prepare file buffer
	m_fileBufferSize = 1024*1024;
	m_fileBuffer     = new BYTE[m_fileBufferSize];

    // setup log
	g_vars.m_log << CLog::Push(new CLog::StdOut(false))
	             << CLog::Push(new CLog::StdErr(true))
	             << CLog::Push(new CLog::FileOut(true));

	int i, j;

	// prepare tearline
	_stprintf( g_vars.m_szTearLine, _T("\r--- %s"), CAdrenalin::getName().c_str() );

	// prepare forward flags
	m_fForwardsRead    = false;
	m_fForwardsChanged = false;

	// keep startup path
	GetModuleFileName( NULL, m_szStartupPath, MAX_PATH );
	for (i = 0, j = -1; m_szStartupPath[i] != _T('\0'); i++) {
	if (m_szStartupPath[i] == _T('\\'))
		j = i;
	}
	m_szStartupPath[++j] = _T('\0');
	// keep startup drive
	lstrcpyn( m_szStartupDrive, m_szStartupPath, 3 );

	// read configuration
	ParseAppCfg();

	try {
		// we initiate load of links and areas
		CLinksManager::getInstance();
		CAreasManager::getInstance();
	} catch (CBadConfigException& e) {
		g_vars.m_log << e.getMessage().c_str() << TEXT_EOL;
		// TODO: ALERT: fuckingbad code found!!! ;)
		m_iErrorCode = ERROR_NONE + 1;
		m_dwAction   = ACTION_ERROR;

		return;
	}

	// possibly now log is working in StdErr+FileOut mode
	// "possibly" means that FileOut may be turned off if we failed to open log file

	// no comments ;)

	if (m_iErrorCode == ERROR_NONE) {
		CheckDiskLimits();
	}
	else {
		TCHAR	szErrorMessage[256];
		GetErrorMessage( m_iErrorCode, szErrorMessage );
		g_vars.m_log << TEXT("Error: ") << szErrorMessage << TEXT_EOL;
	}

	// can we continue?

	if (m_iErrorCode == ERROR_NONE) {
		AssignFileFlags();
	}

	// prepare language
	if (m_iErrorCode == ERROR_NONE) {
		if (_taccess( m_szLngFileName, 0) == 0) {
			if (!m_language.Read(m_szLngFileName)) {
                g_vars.m_log << TEXT("! Error reading language file ") 
					         << m_szLngFileName
                             << TEXT_EOL;
				m_iErrorCode = ERROR_LNG_READ;
				m_dwAction   = ACTION_ERROR;
			}
			else if (m_language.Count() != LNG_COUNT) {
				g_vars.m_log << TEXT("! Wrong number of messages in language file:\n")
					         << TEXT("  ") 
					         << (short int)m_language.Count() 
					         << TEXT(" instead of ") 
					         << LNG_COUNT
                             << TEXT_EOL;
				m_iErrorCode = ERROR_LNG_NUMBER;
				m_dwAction   = ACTION_ERROR;
			}
		}
		else {
			g_vars.m_log << TEXT("! Language file ") << m_szLngFileName 
                         << TEXT(" not found.\n");
			m_iErrorCode = ERROR_LNG_FILE_NOT_FOUND;
			m_dwAction   = ACTION_ERROR;
		}
	}

	if (m_iErrorCode == ERROR_NONE) {
		ParseCmdLine( argc, argv );
	}
	else {
		m_dwAction = ACTION_ERROR;
	}

	if (m_iErrorCode == ERROR_NONE) {
		if (!AllocGlobalMsgBuf()) {
			m_iErrorCode = ERROR_MEMORY;
			m_dwAction   = ACTION_ERROR;
		}
	}

	// restore log
	g_vars.m_log << CLog::Pop(3);
}


CAdrenalin::~CAdrenalin() {
	if (m_fForwardsChanged)
		SaveForwards();

	FreeGlobalMsgBuf();

	// free file buffer
	delete[] m_fileBuffer;

	// we want to print stop time just in log file without stdout and stderr

	SYSTEMTIME	stopTime;
	::GetLocalTime( &stopTime );
	TCHAR	buffer[128];

	formatLogTime( stopTime, buffer );

	g_vars.m_log << CLog::SetSingle(new CLog::FileOut())
		     << TEXT("Stop : ") << buffer << TEXT_EOL
		     << CLog::Pop();
}


void formatSize(DWORD size, char* str) {
	// 1234 -> 1,234

	int	i, j, count;

	count = 0;
	j = 0;
	while (TRUE) {
		str[j] = (char)(size % 10) + '0';
		size /= 10;
		j++;
		count++;
		if (count == 3 && size > 0) {
			str[j] = ',';
			j++;
			count = 0;
		}
		if (size == 0)
			break;
	}
	str[j] = '\0';

	for (i = 0; i < (j >> 1); i++) {
		char	c  = str[i];
		str[i]     = str[j - 1 - i];
		str[j - 1 - i] = c;
	}
}

struct SSize {
	DWORD		m_freeKBytes;
	int		m_iWhere;
};

void CAdrenalin::CheckDiskLimits() {
	// lets get OS version info to decide what function to use:
	//   GetDiskFreeSpace or GetDiskFreeSpaceEx
	bool		fUseExFunction = false;
	OSVERSIONINFO	versionInfo;

	if (GetVersionEx(&versionInfo)) {
		switch (versionInfo.dwPlatformId) {
			case VER_PLATFORM_WIN32_WINDOWS:
				if (LOWORD(versionInfo.dwBuildNumber) > 1000)
					fUseExFunction = true;
				break;
			case VER_PLATFORM_WIN32_NT:
				fUseExFunction = true;
				break;
                }
	}

	BOOL		fSuccess;
	DWORD		dwSectorsPerCluster;
	DWORD		dwBytesPerSector;
	DWORD		dwNumberOfFreeClusters;
	DWORD		dwTotalNumberOfClusters;
	ULARGE_INTEGER	freeBytesAvail;
	ULARGE_INTEGER	totalBytes;
	ULARGE_INTEGER	totalFreeBytes;

	vector<SSize>	vFailedSizes;

	int i = 0;
	for (i = 0; i < m_vDiskLimits.size(); i++) {
		if (fUseExFunction) {
			fSuccess = GetDiskFreeSpaceEx( m_vDiskLimits[i].m_szRoot,
                        			      &freeBytesAvail,
                                                      &totalBytes,
                                                      &totalFreeBytes );
		}
		else {
			fSuccess = GetDiskFreeSpace( m_vDiskLimits[i].m_szRoot,
                        			    &dwSectorsPerCluster,
                                                    &dwBytesPerSector,
                                                    &dwNumberOfFreeClusters,
                                                    &dwTotalNumberOfClusters );
                        if (fSuccess) {
                        	freeBytesAvail.LowPart  = (dwSectorsPerCluster*dwBytesPerSector*dwNumberOfFreeClusters) & 0xFFFFFFFF;
                        	freeBytesAvail.HighPart = (dwSectorsPerCluster*dwBytesPerSector*dwNumberOfFreeClusters) >> 32;
                        }
		}
		if (fSuccess) {
			DWORD	h = freeBytesAvail.HighPart;
			DWORD	l = freeBytesAvail.LowPart;
			DWORD	freeKBytes = ((__int64)l >> 10) + ((__int64)h << 22);
			if (freeKBytes < m_vDiskLimits[i].m_dwLimit) {
				SSize	size;
                                size.m_freeKBytes = freeKBytes;
                                size.m_iWhere   = i;
				vFailedSizes.push_back( size );
			}
		}
		else {
                        TCHAR* lpMsgBuf;
                        
                        FormatMessage(
                            FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
                            NULL,
                            GetLastError(),
                            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
                            (LPTSTR) &lpMsgBuf,
                            0,
                            NULL
                        );
                        
                        // truncate tail CRs and LFs
                        int	len = lstrlen(lpMsgBuf);
                        for (int j = len-1; j > 0; j--) {
                        	if (lpMsgBuf[j] == _T('\r') || lpMsgBuf[i] == _T('\n'))
                        		lpMsgBuf[j] = _T('\0');
                        	else
                        		break;
                        }

                        // Display the string.
                        g_vars.m_log << "! Error getting drive " << m_vDiskLimits[i].m_szRoot << _T(" free space:\r\n") << lpMsgBuf << TEXT_EOL;
                        
                        // Free the buffer.
                        LocalFree( lpMsgBuf );
		}
	}

	if (!vFailedSizes.empty()) {
		g_vars.m_log << TEXT("! Low disk space\n");

		m_iErrorCode = ERROR_LOW_DISK_SPACE;
		TCHAR	szErrorMsg[256];
		TCHAR	szBuf[256];
		vector<SSize>::iterator	it = vFailedSizes.begin();
		while (it != vFailedSizes.end()) {
			formatSize( (*it).m_freeKBytes, szBuf );
			_stprintf( szErrorMsg, _T("  %s -  %s kByte(s) with the limit of "), m_vDiskLimits[(*it).m_iWhere].m_szRoot, szBuf );
			formatSize( m_vDiskLimits[(*it).m_iWhere].m_dwLimit, szBuf );
			_stprintf( szErrorMsg + lstrlen(szErrorMsg), _T("%s KByte(s)\r\n"), szBuf );

			g_vars.m_log << szErrorMsg;
			
			it++;
		}

		// prepare data for sysop
		TCHAR	szLowSpaceFileName[MAX_PATH];
		TCHAR	szCurTime[256];
		char	szOemBuf[256];
		bool	fAppend = false;

                time_t		timestamp = time(NULL);
                struct tm*	tmStruct = localtime(&timestamp);

                _stprintf( szCurTime, _T("%02d-%02d-%04d"), tmStruct->tm_mday, tmStruct->tm_mon + 1, (tmStruct->tm_year < 80 ? tmStruct->tm_year + 2000 : tmStruct->tm_year + 1900) );

		lstrcpy( szLowSpaceFileName, m_szStartupPath );
		lstrcat( szLowSpaceFileName, _T("LowSpace.dat") );

		CharToOem( szLowSpaceFileName, szOemBuf );

		ifstream	in(szOemBuf);
		if (in) {
			
			// read date
                        in.getline( szOemBuf, 256 );

			if (in) {
				OemToChar( szOemBuf, szBuf );
				if (lstrcmp( szBuf, szCurTime ) == 0) {
					fAppend = true;
					// skip all low space data already found
					while (!in.eof() && !vFailedSizes.empty()) {
						in.getline( szOemBuf, 256 );
						if (!in)
							break;
						it = vFailedSizes.begin();
						while (it != vFailedSizes.end() && lstrcmpi( m_vDiskLimits[(*it).m_iWhere].m_szRoot, szBuf ) != 0) {
							it++;
						}
						if (it != vFailedSizes.end()) {
							vFailedSizes.erase(it);
						}
					}
				}
			}
			in.close();
		}
		if (!vFailedSizes.empty()) {
			CFile	out;
			if (fAppend) {
        			out.Open( szLowSpaceFileName, GENERIC_WRITE, OPEN_ALWAYS );
				out.SeekToEnd();
			}
        		else
				out.Open( szLowSpaceFileName, GENERIC_WRITE, CREATE_ALWAYS );
       			out.WriteStringAsOem( szCurTime );
			out.WriteOemString( "\r\n" );

        		vector<tstring>	vBodyLines;

        		vBodyLines.push_back(_T(""));
        		vBodyLines.push_back(_T("Low disk space on drive(s):"));

        		TCHAR	szLine[256];

        		it = vFailedSizes.begin();
        		while (it != vFailedSizes.end()) {
        			formatSize( (*it).m_freeKBytes, szBuf );
        			_stprintf( szLine, _T("  %s - %s kbyte(s) with the limit of "), m_vDiskLimits[(*it).m_iWhere].m_szRoot, szBuf );
        			formatSize( m_vDiskLimits[(*it).m_iWhere].m_dwLimit, szBuf );
        			_stprintf( szLine + lstrlen(szLine), _T("%s kbyte(s)"), szBuf );
        			vBodyLines.push_back( szLine );
        			out.WriteStringAsOem( m_vDiskLimits[(*it).m_iWhere].m_szRoot );
				out.WriteOemString( "\r\n" );
				it++;
        		}
        		out.Close();
                        vBodyLines.push_back( _T("") );
                        _stprintf( szLine, _T("--- %s"), CAdrenalin::getName().c_str() );
                        vBodyLines.push_back( szLine );
        		MakeDirStructure( g_vars.m_szNetmailPath );
        		CMsg	msg;
        		msg.m_addrFrom = g_vars.m_addrMain;
        		msg.m_addrTo   = g_vars.m_addrMain;
        		CharToOem( g_vars.m_szSysOp, msg.m_Header.m_szFromUserName );
        		strcpy( msg.m_Header.m_szToUserName, "Adrenalin" );
                        msg.Reply( ATTR_PRIVATE | ATTR_LOCAL, 0, _T("Low disk space!"), vBodyLines );
		}
	}
}


void CAdrenalin::ParseCmdLine(int argc, TCHAR **argv) {
    int i;

    g_vars.m_log << CLog::StdErr(true);
    
    m_dwAction = ACTION_NONE;

    if (argc <= 1) { // no parameters
		actions.push_back( new CHelpAction(CHelpAction::BRIEF_HELP) );
    }
    else {
		// parse actions
        for (i = 1; i < argc; i++) {
            if (lstrcmpi( argv[i], _T("HELP") ) == 0) {
				actions.push_back( new CHelpAction(CHelpAction::DETAILED_HELP) );
            }
            else if (lstrcmpi( argv[i], _T("FIX") ) == 0) {
                m_dwAction |= ACTION_FIX;
            }
            else if (lstrcmpi( argv[i], _T("TOSS") ) == 0) {
                m_dwAction |= ACTION_TOSS;
            }
			else if (lstrcmpi( argv[i], TEXT("TOSSBAD") ) == 0) {
				m_dwAction |= ACTION_TOSS_BAD;
			}
            else if (lstrcmpi( argv[i], _T("REPORT") ) == 0) {
                m_dwAction |= ACTION_REPORT;
            }
            else if (lstrcmpi( argv[i], _T("DEBUG") ) == 0) {
				m_dwAction |= ACTION_DEBUG;
            }
			else if (lstrcmpi( argv[i], _T("RELINK") ) == 0) {
				m_dwAction |= ACTION_RELINK;
				if (i == argc - 1) {
					g_vars.m_log << TEXT("RELINK address absent") << TEXT_EOL;
					m_dwAction |= ACTION_ERROR;
					m_iErrorCode = ERROR_UNDEFINED_ADDRESS;
					break;
				}
				i++;
				// get relink address
				relinkData.address = argv[i];
				i++;
				// get relink robot name
				if (i < argc) {
					if (lstrlen(argv[i]) > 35) {
						g_vars.m_log << TEXT("Warning: too long robot name, truncated to 35 symbols.") << TEXT_EOL;
					}
					TCHAR	buffer[36];
					lstrcpyn( buffer, argv[i], 36 );
					relinkData.robotName = buffer;
				}
			}
            else if (lstrcmpi( argv[i], _T("PURGE") ) == 0) {
                m_dwAction |= ACTION_PURGE;
                // check if we have areas to purge defined
                if (i < argc-1) {
                    for (++i; i < argc; i++) {
			    m_vsPurgeAreas.push_back( argv[i] );
                    }
                }
            }
			/* 
			 * TimePurge 
			 *
			 */
			else if (lstrcmpi( argv[i], _T("TIMEPURGE") ) == 0) {
				m_dwAction |= ACTION_TIMEPURGE;
				i++;
				if (i == argc) {
					g_vars.m_log << _T("Error: hours for TimePurge not defined\n");
					m_dwAction |= ACTION_ERROR;
					m_iErrorCode = ERROR_TIMEPURGE_NO_HOURS;
					break;
				}
				if (!StrToInt( argv[i], m_timePurge.m_iHours ) 
					|| m_timePurge.m_iHours <= 0)
				{
					g_vars.m_log << _T("Error: invalid hours value\n");
					m_dwAction |= ACTION_ERROR;
					m_iErrorCode = ERROR_TIMEPURGE_INVALID_HOURS;
					break;
				}
				i++;
				while (i < argc) {
					m_timePurge.m_vsMasks.push_back( argv[i] );
					i++;
				}
				break;
			}
			/* 
			 * BasePurge 
			 *
			 */
			else if (lstrcmpi( argv[i], _T("BASEPURGE") ) == 0) {
				m_dwAction |= ACTION_BASEPURGE;
				i++;
				if (i == argc) {
					g_vars.m_log << _T("Error: days for BasePurge not defined\n");
					m_dwAction |= ACTION_ERROR;
					m_iErrorCode = ERROR_BASEPURGE_NO_DAYS;
					break;
				}
				if (!StrToInt( argv[i], m_basePurge.m_iDays ) 
					|| m_basePurge.m_iDays <= 0)
				{
					g_vars.m_log << _T("Error: invalid days value\n");
					m_dwAction |= ACTION_ERROR;
					m_iErrorCode = ERROR_BASEPURGE_INVALID_DAYS;
					break;
				}
				i++;
				while (i < argc) {
					m_basePurge.m_vsAreas.push_back( argv[i] );
					i++;
				}
				break;
			}

			/* 
			 * Hand
			 *
			 */

			else if (lstrcmpi( argv[i], _T("HAND") ) == 0) {
				m_dwAction |= ACTION_HAND;
				i++;
				if (i == argc) {
					g_vars.m_log << _T("Error: link address absent\n");
					m_dwAction |= ACTION_ERROR;
					m_iErrorCode = ERROR_HAND_NO_LINK;
					break;
				}
				m_handCommands.m_sAddr = argv[i];
				i++;
				if (i == argc) {
					g_vars.m_log << _T("Error: no commands were specified\n");
					m_dwAction |= ACTION_ERROR;
					m_iErrorCode = ERROR_HAND_NO_COMMANDS;
					break;
				}
				while (i < argc) {
					m_handCommands.m_vsCommands.push_back( argv[i] );
					i++;
				}
				break;
			}
			// change group
			else if (lstrcmpi( argv[i], TEXT("CHANGEGROUP") ) == 0) {
				i += 2;
				if (i >= argc) {
					g_vars.m_log << TEXT("Error: not enough arguments\n");
					m_dwAction |= ACTION_ERROR;
					m_iErrorCode = ERROR_FEW_ARGUMENTS;
					break;
				}
				actions.push_back( new CChangeGroupAction( argv[i-1], argv[i] ) );
				++i;
				break;
			}
            else if (lstrcmpi( argv[i], _T("HATCH") ) == 0) {
                m_dwAction |= ACTION_HATCH;
                i++;
                if (i == argc) {
                    g_vars.m_log << _T("Error: file to hatch not defined\r\n");
                    m_dwAction |= ACTION_ERROR;
                    m_iErrorCode = ERROR_HATCH_NO_FILE_DEF;
                    break;
                }
                lstrcpy( m_szHatchFile, argv[i] );
                i++;
                if (i == argc) {
                    g_vars.m_log << _T("Error: area to hatch not defined\r\n");
                    m_dwAction |= ACTION_ERROR;
                    m_iErrorCode = ERROR_HATCH_NO_AREA_DEF;
                    break;
                }
                lstrcpy( m_szHatchArea, argv[i] );
                i++;
                if (i == argc) {
                    g_vars.m_log << _T("Error: description not defined\r\n");
                    m_dwAction |= ACTION_ERROR;
                    m_iErrorCode = ERROR_HATCH_NO_DESC_DEF;
                    break;
                }
                lstrcpy( m_szHatchDesc, argv[i] );
                m_szHatchReplaces[0] = _T('\0');
                m_fLeaveHatchedFile = false;
                i++;
                while (i < argc && argv[i][0] == _T('-')) {
                    if (argv[i][1] == _T('R') || argv[i][1] == _T('r')) {
                        // replaces
                        i++;
                        if (i == argc) {
                            g_vars.m_log << _T("Error: Replaces not defined\r\n");
                            m_dwAction |= ACTION_ERROR;
                            m_iErrorCode = ERROR_HATCH_NO_REPLACES;
                            break;
                        }
                        lstrcpy( m_szHatchReplaces, argv[i] );
                        i++;
                    }
                    else if (argv[i][1] == _T('L') || argv[i][1] == _T('l')) {
                    	m_fLeaveHatchedFile = true;
                    	i++;
                    }
                    else {
                        g_vars.m_log << _T("Error: unknown command line option ") << argv[i]+1 << _T("\r\n");
                        m_dwAction |= ACTION_ERROR;
                        m_iErrorCode = ERROR_UNKNOWN_OPT;
                        break;
                    }
                }
                if (i < argc && ((m_dwAction & ACTION_ERROR) == 0)) {
                    g_vars.m_log << _T("Error: extra parameter(s) for hatch\r\n");
                    m_dwAction |= ACTION_ERROR;
                    m_iErrorCode = ERROR_HATCH_EXTRA_ARG;
                }
                break;
            }
            else {
                g_vars.m_log << _T("Error: unknown command line parameter ") << argv[i] << _T("\r\n");
                m_dwAction |= ACTION_ERROR;
                m_iErrorCode = ERROR_UNKNOWN_ARG;
                break;
            }
        }
    }

    g_vars.m_log << CLog::StdErr(false);
}


int CAdrenalin::Execute() {
	while (!actions.empty()) {
		ActionPtr	action = actions.front();
		actions.pop_front();

		try {
			action->invoke();	// can throw CActionException
		} catch(CActionException ae) {
			g_vars.m_log << CLog::ResetSingle(new CLog::StdOut()) 
						 << TEXT("Exception occured: ")
						 << ae.getMessage() << TEXT_EOL << CLog::Pop(3);
		}

		delete action;
	}

	if (m_dwAction & ACTION_ERROR) {		// prevously failed
		return (m_iErrorCode);
	}
	if (m_dwAction & ACTION_FIX) {		// fix
		ProcessFixRequests();
	}
	if (m_dwAction & ACTION_TOSS) {		// toss
		ProcessInboundTics( g_vars.m_szInboundPath, g_vars.m_szInboundPath );
	}
	if (m_dwAction & ACTION_TOSS_BAD) {		// toss
		tossingBadTics = true;
		ProcessInboundTics( g_vars.m_szBadTicsPath, g_vars.m_szInboundPath );
		tossingBadTics = false;
	}
	if (m_dwAction & ACTION_RELINK) {		// relink
		return Relink();
	}
	if (m_dwAction & ACTION_PURGE) {		// purge
		Purge();
	}
	if (m_dwAction & ACTION_HATCH) {		// hatch
		return (Hatch());
	}
	if (m_dwAction & ACTION_REPORT) {		// report
		g_vars.m_log << _T("REPORT under construction\r\n");
	}
	if (m_dwAction & ACTION_DEBUG) {		// debug
		PrintDebugInfo();
	}
	if (m_dwAction & ACTION_TIMEPURGE) {	// TimePurge
		TimePurge();
	}
	if (m_dwAction & ACTION_BASEPURGE) {	// purge based on data from files.db
		BasePurge();
	}
	if (m_dwAction & ACTION_HAND) {	// Hand
		ProcessFixRequests(true);
	}
	return (0);
}


bool CAdrenalin::ParseLinkAddr(LPCTSTR s, SFTNAddress &Addr) {
    TCHAR	szFullAddr[24+MAX_DOMAIN_SIZE];
    TCHAR	szTempStr[24+MAX_DOMAIN_SIZE];
    TCHAR	szDomain[MAX_DOMAIN_SIZE];
    bool	fColonPresent, fSlashPresent, fPointPresent, fDogPresent;
    int		DogPos;
    int		i;

    if (lstrlen(s) >= 24+MAX_DOMAIN_SIZE)
        return (false);
    
    fColonPresent = fSlashPresent = fPointPresent = fDogPresent = false;
    i = 0;
    while (s[i] != _T('\0')) {
        switch (s[i]) {
        case _T(':'):
            fColonPresent = true;
            break;
        case _T('/'):
            fSlashPresent = true;
            break;
        case _T('.'):
            fPointPresent = true;
            break;
        case _T('@'):
            if (!fDogPresent) {
                fDogPresent = true;
                DogPos      = i;
            }
            break;
        }
        i++;
    }
    lstrcpy( szTempStr, s );
    if (fDogPresent && !fPointPresent) {
        i = lstrlen( szTempStr + DogPos + 1);
        if (i > MAX_DOMAIN_SIZE-1)
            i = MAX_DOMAIN_SIZE-1;
        lstrcpyn( szDomain, szTempStr + DogPos + 1, i+1 );
        szTempStr[DogPos] = _T('\0');
    }
    if (!fSlashPresent) {
        _stprintf( szFullAddr, _T("%lu/%s"), g_vars.m_addrMain.Net, szTempStr );
        lstrcpy( szTempStr, szFullAddr );
    }
    if (!fColonPresent) {
        _stprintf( szFullAddr, _T("%lu:%s"), g_vars.m_addrMain.Zone, szTempStr );
    }
    else
        lstrcpy( szFullAddr, szTempStr );

    if (!ParseFTNAddress( szFullAddr, Addr ))
        return (false);
    
    if (!fDogPresent) {
        lstrcpy( Addr.Domain, g_vars.m_addrMain.Domain );
    }
    else if (!fPointPresent) {
        lstrcpy( Addr.Domain, szDomain );
    }

    return (*(Addr.Domain) != _T('\0'));
}


/*
bool CAdrenalin::ParseAreaAccess(LPCTSTR s, SArea &Area) {
    int  i;

    i = 0;
    Area.m_nPriority = 0;
    while (_istdigit(*s) && i < 4) {
        Area.m_nPriority *= 10;
        Area.m_nPriority += *s - _T('0');
        s++;
        i++;
    }
    
    if (Area.m_nPriority > 1024)
        return (false);
    
    if (*s == _T('\0')) {
        Area.m_ulGroup = 0;
        return (true);
    }
    
    if (*s != _T('/'))
        return (false);
    
    s++;
    
    if (!_istalpha(*s))
        return (false);
    
    Area.m_ulGroup |= 1 << (ToUpper(*s) - _T('A'));
    
    s++;
    
    return (*s == _T('\0'));
}
*/


void CAdrenalin::MoveTicToBads(LPCTSTR szFullTicName, int TicNameFrom) {
    TCHAR szDestTicName[MAX_PATH];
    TCHAR szLogLine[LINE_SIZE];
    int  i, j, Len;
    bool	fNameFound;
    bool	fNameChanged;

    lstrcpy( szDestTicName, g_vars.m_szBadTicsPath );
    lstrcat( szDestTicName, szFullTicName + TicNameFrom );

    Len = lstrlen(szDestTicName);

    fNameFound   = false;
    fNameChanged = false;
    
    do {

        // trying to check the file
        if (_taccess( szDestTicName, 0) != 0)
            fNameFound = true;
        else {
            
            // form next name
            i = Len-1;
            j = 0;
        
            while (j <= 2) {
                if (szDestTicName[i] < _T('0')) {
                    szDestTicName[i] = _T('0');
                    break;
                }
            
                if (szDestTicName[i] < _T('9')) {
                    szDestTicName[i]++;
                    break;
                }
            
                if (szDestTicName[i] < _T('A')) {
                    szDestTicName[i] = _T('a');
                    break;
                }
                
                if (szDestTicName[i] < _T('Z')) {
                    szDestTicName[i]++;
                    szDestTicName[i] += _T('a') - _T('A');
                    break;
                }
            
                if (szDestTicName[i] < _T('a')) {
                    szDestTicName[i] = _T('a');
                    break;
                }
            
                if (szDestTicName[i] < _T('z')) {
                    szDestTicName[i]++;
                    break;
                }
        
                szDestTicName[i] = _T('0');
                i--;
                j++;
            }

            // check if failed to form new name
            if (j > 2)
                break;
            fNameChanged = true;
        }
    } while (!fNameFound);

    if (!fNameFound || MyMoveFile( szFullTicName, szDestTicName ) != FO_SUCCEEDED)
        lstrcpy( szLogLine, _T("                failed to move TIC into BadTics\r\n") );
    else {
        if (fNameChanged) {
            _stprintf( szLogLine, _T("                TIC was renamed to %s and moved into BadTics\r\n"),
                     szDestTicName + TicNameFrom + Len - lstrlen(szFullTicName) );
        }
        else
            lstrcpy( szLogLine, _T("                TIC was moved into BadTics\r\n") );
    }
    
    g_vars.m_log << szLogLine;
}


int CAdrenalin::Hatch() {

	// check hatch file
	if (m_szHatchFile[0] != TEXT('@')) {
		// we have only one file to hatch
		return HatchSingleFile( m_szHatchFile, m_szHatchDesc );
	}

	// we have filelist

	g_vars.m_log << CLog::Push(new CLog::StdOut(true))
				 << CLog::Push(new CLog::FileOut(true));

	g_vars.m_log << "Executing filelist hatch..." << TEXT_EOL;

	// remember filelist name
	tstring	filelistName( m_szHatchFile+1 );
	// prepare to read lines
	COemLineReader	lineReader(filelistName.c_str());
	if (lineReader.isOpened()) {
		// read filelist and hatch every file

		int	totalFilesCount   = 0;
		int	hatchedFilesCount = 0;

		while (lineReader.next()) {
			// get next line
			LPCTSTR	line = lineReader.getLine();
			int		i, j;
			// extract file name and description
			// file name can be enclosed in double quotes
			TCHAR	separator1, separator2;
			if (line[0] == TEXT('"')) {
				// we suppose double quoted file name
				separator1 = TEXT('"');
				separator2 = ZERO_CHAR;
				// start loop from the second character
				i = 1;
			}
			else {
				// we suppose that file name is not double quoted
				separator1 = SPACE_CHAR;
				separator2 = TAB_CHAR;
				// start loop from the first character
				i = 0;
			}
			j = i;
			while (line[i] != separator1 && line[i] != separator2 && line[i] != ZERO_CHAR) {
				i++;
			}
			if (i > j) {

				// we have file name

				totalFilesCount++;
				tstring	fileName( line + j, i-j );

				// extract description

				if (line[i] != ZERO_CHAR) {
					// skip any separator
					i++;
				}
				// skip leading spaces and tabs
				while (line[i] == SPACE_CHAR || line[i] == TAB_CHAR) {
					i++;
				}
				j = i;
				// skip tailing spaces and tabs
				int	firstSpaceInSeq = -1;
				while (line[i] != ZERO_CHAR) {
					if (line[i] == SPACE_CHAR || line[i] == TAB_CHAR) {
						// we are on space or tab
						if (firstSpaceInSeq < 0) {
							// set space sequence start
							firstSpaceInSeq = i;
						}
					}
					else {
						// set space sequence start
						firstSpaceInSeq = -1;
					}
					i++;
				}
				if (firstSpaceInSeq >= 0)
					i = firstSpaceInSeq;
				tstring	fileDesc;
				if (i > j) {
					// we have description
					fileDesc = tstring( line + j, i-j );
				}

				// now we have file name and file description (description can be empty)
				// check description length (compare with max tic line size not including "DESC ")
				if (fileDesc.size() > (MAX_TIC_LINE_SIZE - 5)) {
					g_vars.m_log << "Warning: description too long, truncated\n";

					fileDesc.resize( MAX_TIC_LINE_SIZE - 5 );
				}

				// if description is empty use description from command line
				if (fileDesc.size() == 0)
					fileDesc = m_szHatchDesc;

				// do hatch file!
				if (HatchSingleFile( fileName.c_str(), fileDesc.c_str() ) == ERROR_NONE)
					hatchedFilesCount++;
			}
		}
		if (totalFilesCount > 0) {
			// log total and hatched files count (and show counts to sysop)
			g_vars.m_log << TEXT("Hatch result: [")
						 << hatchedFilesCount << TEXT("/") << totalFilesCount
						 << TEXT("]") << TEXT_EOL;
		}
		else {
			// warn user about empty file (and log this information)
			g_vars.m_log << TEXT("There are no files found to hatch") << TEXT_EOL;
		}
	}
	else {
		// notify user about failure and log the failure
		g_vars.m_log << TEXT("Failed to read ")
			         << filelistName.c_str()
					 << TEXT_EOL;
	}

	g_vars.m_log << CLog::Pop() << CLog::Pop();

	return ERROR_NONE;
}


void CAdrenalin::AddStartupPath(LPTSTR szLocation) {
	TCHAR	szTempPath[MAX_PATH];

	if (szLocation[0] == _T('\0') || szLocation[1] != _T(':')) {
		lstrcpy( szTempPath, szLocation );
		ConcatStrings( szLocation, m_szStartupPath, szTempPath );
	}
}


CAdrenalin& CAdrenalin::getInstance() {
	return (*pAdrenalin);
}



typedef	vector<PLink>	VPLink;

struct SLinksFile {
	tstring	m_sCfgFileName;
	VPLink	m_vpLink;
};

typedef vector<SLinksFile>	VLinksFile;

bool CAdrenalin::SaveCfg() {
	throw CRuntimeException( "CAdrenalin::SaveCfg should be removed" );
	/*
	VLinksFile	vLinksFile;

	SLink*		pLink;

	ofstream      out;
	ifstream      in;

	TCHAR         szTempFileName[MAX_PATH];

	int           nLine;

	bool          fCompareNext;
	char          c, firstc;
	unsigned long bit;

	VLinksFile::iterator	itItem;
	VPLink::iterator	itLink;
	VPLink::iterator	itLinkNext;

	// prepare temp file name

    	ConcatStrings( szTempFileName, m_szStartupPath, _T("adrenalin.tmp") );
    	if (!GetTmpFileName(szTempFileName))
        	return (false);

        // prepare links file list

        if (m_links.SelectFirst()) {
		do {
        		// search link cfg file name in file list
			itItem = vLinksFile.begin();
        		while (itItem != vLinksFile.end() && lstrcmp( (*itItem).m_sCfgFileName.c_str(), m_links.GetSelected().getCfgFileName().c_str() ) != 0)
        			itItem++;
        		// if corresponding item wasn't found we shell add new one
        		if (itItem == vLinksFile.end()) {
				SLinksFile	linksFile;
				linksFile.m_sCfgFileName = m_links.GetSelected().getCfgFileName();
				linksFile.m_vpLink.push_back( &m_links.GetSelected() );
				vLinksFile.push_back( linksFile );
        		}
        		else {
        			// lets insert link into correct position based on line number
				itLink     = (*itItem).m_vpLink.begin();
				if (itLink == (*itItem).m_vpLink.end()) {
					(*itItem).m_vpLink.push_back(&m_links.GetSelected());
				}
				else {
					itLinkNext = (*itItem).m_vpLink.begin();
					itLinkNext++;
					while ((*itLink)->getLine() < m_links.GetSelected().getLine() && itLinkNext != (*itItem).m_vpLink.end()) {
        					itLink++;
						itLinkNext++;
					}
        				// if linkItem->link->Line less than current link line than
        				// it means that this item is the last item in list
					pLink = &m_links.GetSelected();
        				if ((*itLink)->getLine() < m_links.GetSelected().getLine()) {
							(*itItem).m_vpLink.insert( itLinkNext, pLink );
        				}
        				else {
							(*itItem).m_vpLink.insert( itLinkNext, pLink );
        				}
				}
        		}
		} while (m_links.SelectNext());
	}

        // go through file list and rewrite all files

	char	szOemFileName[MAX_PATH];
	char	szOemBuf[LINE_SIZE];
	char	szOemLine[LINE_SIZE];

        itItem = vLinksFile.begin();
        while (itItem != vLinksFile.end()) {
        	do {
        		// open input and output files
			CharToOem( (*itItem).m_sCfgFileName.c_str(), szOemFileName );
                	in.open( szOemFileName );
                	if (!in)
                		break;
			CharToOem( szTempFileName, szOemFileName );
                	out.open( szOemFileName );
                	if (!out) {
                		in.close();
                		break;
                	}
                	// setup links
                	itLink = (*itItem).m_vpLink.begin();                        
                	nLine = 0;
                	// read, write and rewritelines
			while (!in.eof() && itLink != (*itItem).m_vpLink.end()) {
				pLink = (*itLink);
                                in.getline( szOemLine, LINE_SIZE );
                                if (!in)
                                	continue;
                                nLine++;
                                if (nLine < pLink->getLine())
                                    	out << szOemLine << endl;
                                else {
                                        out << "Link " << setw(24) << setiosflags(ios::left);
                                        // address
                                        *szOemBuf = '\0';
                                        if (lstrcmpi( g_vars.m_addrMain.Domain, pLink->getAddress().Domain ) == 0) {
                                         	fCompareNext = true;
	                                        if (g_vars.m_addrMain.Zone != pLink->getAddress().Zone) {
        	                                	sprintf( szOemBuf, "%lu:", pLink->getAddress().Zone );
                	                        	fCompareNext = false;
                        	                }
                                                if (!fCompareNext || g_vars.m_addrMain.Net != pLink->getAddress().Net)
                                                	sprintf( szOemBuf + strlen(szOemBuf), "%lu/", pLink->getAddress().Net );
                                                sprintf( szOemBuf + strlen(szOemBuf), "%lu", pLink->getAddress().Node );
                                                if (pLink->getAddress().Point != 0)
                                                	sprintf( szOemBuf + strlen(szOemBuf), ".%lu", pLink->getAddress().Point );
                                        }
                                	else
                                    		sprintf( szOemBuf, "%lu:%lu/%lu.%lu@%s", pLink->getAddress().Zone, pLink->getAddress().Net, pLink->getAddress().Node, pLink->getAddress().Point, pLink->getAddress().Domain );
                                	out << szOemBuf;
                                	// password
									CharToOem( pLink->getPassword().c_str(), szOemBuf );
                                	out << " " << setw(15) << setiosflags(ios::left) << szOemBuf << " ";
                                	// route type
                                	switch (pLink->getRouteType()) {
                                                case ROUTE_SEND:
                                                        out << "S";
                                                        break;
                                                case ROUTE_RECEIVE:
                                                        out << "R";
                                                        break;
                                                case ROUTE_BIDIRECT:
                                                        out << "B";
                                                        break;
                                        }
                                        // BSO type
                                        out << " ";
                                        switch (pLink->getBSOType()) {
                                        case BSO_NORMAL:
                                                out << "N";
                                                break;
                                        case BSO_DIRECT:
                                                out << "D";
                                                break;
                                        case BSO_HOLD:
                                                out << "H";
                                                break;
                                        case BSO_CRASH:
                                                out << "C";
                                                break;
                                        case BSO_IMMEDIATE:
                                                out << "I";
                                                break;
                                        }
                                        out << " ";
                                        // allow area creation
                                        if (pLink->getAllowAreaCreation())
                                                out << "A ";
                                        // passive
                                        switch (pLink->getActivity()) {
                                        case ACTIVITY_PASSIVE:
                                            out << "P ";
                                            break;
                                        case ACTIVITY_UNAVAILABLE:
                                            out << "U ";
                                            break;
                                        }
                                        // priority
                                        out << pLink->getPriority() << "/";
                                        // groups
                                        firstc = '\0';
                                        c      = 'A';
                                        bit    = 1;
                                        while (c <= 'Z') {
                                                if ((pLink->getGroups() & bit) != 0) {
                                                        if (firstc == '\0')
                                                                firstc = c;
                                                }
                                                else {
                                                        if (firstc != '\0') {
                                                                out << firstc;
                                                                if (firstc != c-1)
                                                                        out << "-" << (char)(c-1);
                                                                firstc = '\0';
                                                        }
                                                }
                                                c++;
                                                bit <<= 1;
                                        }
                                        if (firstc != '\0') {
                                                if (firstc == 'A')
                                                        out << "*";
                                                else {
                                                        out << firstc;
                                                        if (firstc != c-1)
                                                                out << "*";
                                                }
                                        }
                                        out << endl;
					itLink++;
                        	}
			}
                        while (!in.eof()) {
                                in.getline( szOemLine, LINE_SIZE );
                                if (!in)
                                        continue;
                                out << szOemLine << endl;
                        }
                        in.close();
                        out.close();
                        _tunlink( (*itItem).m_sCfgFileName.c_str() );
                        _trename( szTempFileName, (*itItem).m_sCfgFileName.c_str() );
                } while (false);
		itItem++;
        }

        return (true);
	*/
}


struct	SAreaFiles {
	TCHAR		m_szAreaName[AREA_NAME_SIZE];
	VTString	m_vFiles;
};


void CAdrenalin::Purge() {
	WIN32_FIND_DATA		FindFileData;
	HANDLE			hFindFile;
	BOOL			fFileFound;
	TCHAR			szFindFileMask[MAX_PATH];
	TCHAR			szFullFileName[MAX_PATH];
	TCHAR			szFileName[MAX_PATH];
	int			iNamePos;
	CTic			tic;
	vector<SAreaFiles>	vAreaFiles;
	VTString		vCurFile;
	ifstream		in;
	ofstream		tempout;
	TCHAR			szTempFileName[MAX_PATH];
	TCHAR			szBuf[LINE_SIZE];
	char			szOemBuf[LINE_SIZE];
	int			i;
	int				nPurgedNumber, nTotalPurgedNumber;
	int				nPurgedSize,   nTotalPurgedSize;
	bool				fSkipLongDesc;
	VTString::iterator		itsCurPurgeArea;
	bool				fHaveToAddFile;
	bool				fAreaAsMask;
	vector<SAreaFiles>::iterator	itsArea;

	g_vars.m_log << CLog::StdOut(true) << _T("Purge:\n");

	if (!m_vsPurgeAreas.empty()) {
		for (itsCurPurgeArea = m_vsPurgeAreas.begin(); itsCurPurgeArea != m_vsPurgeAreas.end();
		     itsCurPurgeArea++)
		{
			if ((*itsCurPurgeArea).c_str()[0] == _T('-')) { // exclude from arealist
				itsArea = vAreaFiles.begin();
				while (itsArea != vAreaFiles.end()) {
					if (CheckMask( (*itsArea).m_szAreaName,
							(TCHAR*)(*itsCurPurgeArea).c_str() + 1 ))
					{
						// remove area from list
						vAreaFiles.erase( itsArea );
						itsArea = vAreaFiles.begin();
					}
					else
						itsArea++;
				}
			}
			else { // add to arealist, if one or more areas meet mask
				i = 0; // number of added areas
				fAreaAsMask = IsMask( (*itsCurPurgeArea).c_str() );
				int	areasCount = CAreasManager::getInstance()->getAreasCount();
				for (i = 0; i < areasCount; i++) {
					PArea	area = CAreasManager::getInstance()->getArea( i );
					if (fAreaAsMask) {
						if (area->hasFlag( CArea::FLAG_VETO_MANUAL_PURGE ))
							continue;
						if (!CheckMask( area->getName().c_str(),
							(TCHAR*)(*itsCurPurgeArea).c_str()))
							continue;
					}
					else 
						if (stricmp( area->getName().c_str(),
							(*itsCurPurgeArea).c_str()) != 0)
							continue;
					// add only unique areas
					vector<SAreaFiles>::iterator	itsArea = vAreaFiles.begin();
					while (itsArea != vAreaFiles.end()) {
						if (stricmp( (*itsArea).m_szAreaName, area->getName().c_str() ) == 0)
							break;
						itsArea++;
					}
					if (itsArea == vAreaFiles.end()) { // not found
						SAreaFiles	newAreaFiles;
						strcpy( newAreaFiles.m_szAreaName,
								area->getName().c_str() );
						vAreaFiles.push_back( newAreaFiles );
						i++;
					}
				}
				if (i == 0) { // no areas added
					if (fAreaAsMask)
						_stprintf( szBuf,
							_T("! No purgable areas found for mask %s\n"),
							(*itsCurPurgeArea).c_str() );
					else
						_stprintf( szBuf, _T("! Area %s not found.\n"),
							(*itsCurPurgeArea).c_str() );
					g_vars.m_log << szBuf;
				}
			}
		}
	}
    
	lstrcpy( szFindFileMask, g_vars.m_szNewTicsPath );
	lstrcat( szFindFileMask, _T("*.ti?") );

	hFindFile = ::FindFirstFile( szFindFileMask, &FindFileData );
	fFileFound = (hFindFile != INVALID_HANDLE_VALUE);
	lstrcpy( szFullFileName, g_vars.m_szNewTicsPath );
	iNamePos = lstrlen( szFullFileName );

	while (fFileFound) {
		lstrcpy( szFullFileName + iNamePos, FindFileData.cFileName );
		if (tic.Read( szFullFileName )) {
			fHaveToAddFile = false;
			do {
				PArea	area = CAreasManager::getInstance()->getArea( tic.m_szArea );
				if (area == NULL)
					break;
				// try to find area
				itsArea = vAreaFiles.begin();
				while (itsArea != vAreaFiles.end()) {
					if (lstrcmpi( (*itsArea).m_szAreaName, tic.m_szArea) == 0)
						break;
					itsArea++;
				}
				if (itsArea != vAreaFiles.end()) { // area found
					fHaveToAddFile = true;
					break;
				}
				if (m_vsPurgeAreas.empty() && area->hasFlag( CArea::FLAG_PASSTHROUGH ))
				{
					// we will add area
					fHaveToAddFile = true;
					break;
				}
				break;
			} while (true);
			if (fHaveToAddFile) {
				// prepare file name
				if (*tic.m_szFullName != _T('\0'))
					lstrcpy( szFileName, tic.m_szFullName );
				else
					lstrcpy( szFileName, tic.m_szFile );
				// add file into area files list,
				// with this purpose find area or add it, if it's absent
				if (itsArea == vAreaFiles.end()) {
					// add area
					SAreaFiles	newAreaFiles;
					lstrcpy( newAreaFiles.m_szAreaName, tic.m_szArea );
					// add file
					newAreaFiles.m_vFiles.push_back( szFileName );
					vAreaFiles.push_back( newAreaFiles );
				}
				else {
					// try to find file to exclude duplicates
					VTString::iterator	itsCurFile = (*itsArea).m_vFiles.begin();
					while (itsCurFile != (*itsArea).m_vFiles.end()) {
						if (lstrcmpi( (*itsCurFile).c_str(), szFileName ) == 0)
							break;
						itsCurFile++;
					}
					if (itsCurFile == (*itsArea).m_vFiles.end()) {
						// add file
						(*itsArea).m_vFiles.push_back( szFileName );
					}
				}
			}
		}
		fFileFound = ::FindNextFile( hFindFile, &FindFileData );
	}
	if (hFindFile != INVALID_HANDLE_VALUE)
		::FindClose( hFindFile );

	// add passthrough areas if not directly specified
	if (m_vsPurgeAreas.empty()) {
		int	areasCount = CAreasManager::getInstance()->getAreasCount();
		for (int i = 0; i < areasCount; i++) {
			PArea	area = CAreasManager::getInstance()->getArea( i );
			if (area->hasFlag( CArea::FLAG_PASSTHROUGH )) {
				itsArea = vAreaFiles.begin();
				while (itsArea != vAreaFiles.end()) {
					if (lstrcmpi( (*itsArea).m_szAreaName, area->getName().c_str() ) == 0)
						break;
					itsArea++;
				}
				if (itsArea == vAreaFiles.end()) {
					// add area
					SAreaFiles	newAreaFiles;
					lstrcpy( newAreaFiles.m_szAreaName, area->getName().c_str() );
					vAreaFiles.push_back( newAreaFiles );
				}
			}
		}
	}

	// go through all selected areas
	nTotalPurgedSize   = 0;
	nTotalPurgedNumber = 0;
	itsArea      = vAreaFiles.begin();
	while (itsArea != vAreaFiles.end()) {
		_stprintf( szBuf, _T("* Area: %s\n"), (*itsArea).m_szAreaName );
		g_vars.m_log << szBuf;
		// select area
		PArea	area = CAreasManager::getInstance()->getArea( (*itsArea).m_szAreaName );
		// prepare directory
		lstrcpy( szFullFileName, area->getWorkingPath().c_str() );
		iNamePos = lstrlen( szFullFileName );
		// prepare mask to look files
		lstrcpy( szFindFileMask, area->getWorkingPath().c_str() );
		lstrcat( szFindFileMask, _T("*.*") );
		// process fecho files
		nPurgedNumber = 0; // number of removed files
		nPurgedSize   = 0; // size of purged files in this area
		hFindFile    = FindFirstFile( szFindFileMask, &FindFileData );
		fFileFound    = (hFindFile != INVALID_HANDLE_VALUE);
		while (fFileFound) {
			// skip directories, special files and "files.bbs"
			if (!(FindFileData.dwFileAttributes &
				(FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_HIDDEN |
				FILE_ATTRIBUTE_OFFLINE | FILE_ATTRIBUTE_READONLY |
				FILE_ATTRIBUTE_TEMPORARY))
				&&
				lstrcmpi( FindFileData.cFileName, g_vars.m_szFilelistFileName ) != 0)
			{
				// let's check, if we have TIC for this file
				VTString::iterator	itsCurFile = (*itsArea).m_vFiles.begin();
				while (itsCurFile != (*itsArea).m_vFiles.end()) {
					if (lstrcmpi( FindFileData.cFileName, (*itsCurFile).c_str()) == 0)
						break;
					itsCurFile++;
				}
				// remove file, if it's not used
				if (itsCurFile == (*itsArea).m_vFiles.end()) {
					// form full name
					lstrcpy( szFullFileName + iNamePos, FindFileData.cFileName );
					if (_tunlink( szFullFileName ) == 0) {
						_stprintf( szBuf, _T("        Removed file %s\n"),
							FindFileData.cFileName );
						nPurgedNumber++;
						nPurgedSize += (FindFileData.nFileSizeHigh * MAXDWORD) +
							FindFileData.nFileSizeLow;
					}
					else
						_stprintf( szBuf, _T("        Error removing file %s\n"),
							FindFileData.cFileName );
					g_vars.m_log << szBuf;
				}
			}
			fFileFound = FindNextFile( hFindFile, &FindFileData );
		}
		if (hFindFile != INVALID_HANDLE_VALUE)
			FindClose( hFindFile );

		_stprintf( szBuf, _T("        %ld file(s) of size "), nPurgedNumber );
		FormatSize( nPurgedSize, szBuf + lstrlen(szBuf) );
		lstrcat( szBuf, _T(" byte(s).\n") );
		g_vars.m_log << szBuf;
		// change "files.bbs"
		// set file names
		lstrcpy( szFullFileName + iNamePos, g_vars.m_szFilelistFileName );
		lstrcpy( szTempFileName, area->getWorkingPath().c_str() );
		lstrcat( szTempFileName, _T("adrenalin.tmp") );
		// check names for spaces
		VTString::iterator	itsCurFile = (*itsArea).m_vFiles.begin();
		while (itsCurFile != (*itsArea).m_vFiles.end()) {
			for (i = 0; i < (*itsCurFile).length(); i++) {
				if ((*itsCurFile).c_str()[i] == _T(' '))
					break;
			}
			if ((*itsCurFile).c_str()[i] != _T('\0')) {
				lstrcpy( szFileName, _T("\"") );
				lstrcat( szFileName, (*itsCurFile).c_str() );
				lstrcat( szFileName, _T("\"") );
				(*itsCurFile) = szFileName;
			}
			itsCurFile++;
		}
		
		// copy desc file into temp file, but skip lines with absent files
		bool	fOverwriteSuccess = true;
		do {
			// check temp file name
			if (!GetTmpFileName( szTempFileName ))
				break;
			CharToOem( szFullFileName, szOemBuf );
			in.open( szOemBuf );
			if (!in)
				break;
			CharToOem( szTempFileName, szOemBuf );
			tempout.open( szOemBuf );
			if (!tempout) {
				in.close();
				break;
			}
			fSkipLongDesc = false;
			while (!in.eof()) {
				in.getline( szOemBuf, LINE_SIZE );
				if (!in) {
					if (!in.eof())
						break;
					continue;
				}
				OemToChar( szOemBuf, szBuf );
				// check file name
				itsCurFile = (*itsArea).m_vFiles.begin();
				while (itsCurFile != (*itsArea).m_vFiles.end()) {
					if (_tcsnicmp( szBuf, (*itsCurFile).c_str(),
						(*itsCurFile).length()) == 0)
						break;
					itsCurFile++;
				}
				if (itsCurFile == (*itsArea).m_vFiles.end())
					fSkipLongDesc = true;
				else {
					if (fSkipLongDesc) {
						if (*szBuf != _T(' ') && *szBuf != _T('>') && *szBuf != _T('+')) {
							fSkipLongDesc = false;
							tempout << szOemBuf << endl;
							if (!tempout) {
								fOverwriteSuccess = false;
								break;
							}
						}
					}
					else {
						tempout << szOemBuf << endl;
						if (!tempout) {
							fOverwriteSuccess = false;
							break;
						}
					}
				}
			}
			in.close();
			tempout.close();
			if (fOverwriteSuccess) {
				_tunlink( szFullFileName );
				_trename( szTempFileName, szFullFileName );
			}
			else {
				_tunlink( szTempFileName );
				g_vars.m_log << _T("! error updating file ") << szFullFileName << _T("\n");
			}
			break;
		} while (true);
		// remove "files.bbs", if it's empty
		hFindFile = FindFirstFile( szFullFileName, &FindFileData );
		fFileFound = (hFindFile != INVALID_HANDLE_VALUE);
		if (fFileFound) {
			FindClose( hFindFile );
			if (((FindFileData.nFileSizeHigh * MAXDWORD) +
				FindFileData.nFileSizeLow) == 0)
			_tunlink( szFullFileName);
		}
		nTotalPurgedNumber += nPurgedNumber;
		nTotalPurgedSize   += nPurgedSize;
		// free item memory
		(*itsArea).m_vFiles.clear();
		itsArea++;
	}
    
	_stprintf( szBuf, _T("Total: %ld file(s) of size "), nTotalPurgedNumber );
	FormatSize( nTotalPurgedSize, szBuf + lstrlen(szBuf) );
	lstrcat( szBuf, _T(" byte(s).\n") );
	g_vars.m_log << szBuf;

	g_vars.m_log << CLog::StdOut(false);

}

int CAdrenalin::Relink() {
	CMsg		msg;
	VTString	vBodyLines;
	TCHAR		szLine[256];
	int		nAreas;

	// prepare link address;
	SFTNAddress	relinkAddress;
	if (!ParseLinkAddr( relinkData.address.c_str(), relinkAddress )) {
		g_vars.m_log << CLog::StdErr(true) 
			     << TEXT("Failed to parse RELINK address ")
			     << relinkData.address
			     << TEXT_EOL
			     << CLog::StdErr(false);
		return ERROR_WRONG_ADDRESS;
	}

	relinkAddress.ToString( szLine );

	// check link
	PLink	link = CLinksManager::getInstance()->getLink(relinkAddress);
	if (link == NULL) {
		g_vars.m_log << CLog::StdErr(true) 
			     << TEXT("Relink failed because of unknown link ")
			     << szLine << TEXT_EOL
			     << CLog::StdErr(false);
		return ERROR_WRONG_ADDRESS;
	}

	g_vars.m_log << CLog::StdOut(true);

	// print log message
	g_vars.m_log << TEXT("Relink: ") << szLine << TEXT_EOL;

	nAreas = 0;
	int	areasCount = CAreasManager::getInstance()->getAreasCount();
	for (int i = 0; i < areasCount; i++) {
		PArea	area = CAreasManager::getInstance()->getArea( i );
		TiePtr	tie  = area->getTie( link );
		if (tie != NULL) {
			// add msg line
			_stprintf( szLine, TEXT("+%s"), area->getName().c_str() );
			vBodyLines.push_back( szLine );
			nAreas++;
			// print log message
			g_vars.m_log << TEXT("  ") << szLine << TEXT_EOL;
		}
	}

	g_vars.m_log << nAreas << " area(s) added into the relink request." << TEXT_EOL;

	if (nAreas > 0) {
		vBodyLines.push_back( TEXT("") );
		MakeDirStructure( g_vars.m_szNetmailPath );
		// from/to fields reversed because we will use reply method
		msg.m_addrFrom = link->getAddress();
		msg.m_addrTo   = link->getOurAddress();
		// check distant robot name
		if (relinkData.robotName.size() == 0) {
			if (link->getRobotName().length() == 0)
				strcpy( msg.m_Header.m_szFromUserName, "AllFix" );
			else
				CharToOem( link->getRobotName().c_str(), msg.m_Header.m_szFromUserName );
		}
		else {
			CharToOem( relinkData.robotName.c_str(), msg.m_Header.m_szFromUserName );
		}
		CharToOem( g_vars.m_szSysOp, msg.m_Header.m_szToUserName );
		msg.Reply( ATTR_PRIVATE | ATTR_LOCAL, 0, link->getPassword().c_str(), vBodyLines );
		g_vars.m_log << TEXT("Relink request message created.")
			     << TEXT_EOL;
	}
	else {
		g_vars.m_log << TEXT("There is no need to create relink request message because of no areas linked.")
			     << TEXT_EOL;
	}

	g_vars.m_log << CLog::StdOut(false);

	return ERROR_NONE;
}

SFTNAddress& CAdrenalin::GetOurAddr(int iOurAddr) {
	if (iOurAddr == MAIN_ADDRESS)
		return (g_vars.m_addrMain);
	return (g_vars.m_vAKAAddr[iOurAddr]);
}

void CAdrenalin::LoadForwards() {
	int		i;
	TCHAR		szFileName[MAX_PATH];
	LPTSTR		szFrom;
	SForwardInfo	fwdInfo;
	int		nCurrentDays;

	// 0. Clear vector and prepare time
	m_vectForwardInfo.clear();
	SYSTEMTIME	systemTime;
	GetLocalTime( &systemTime );
	nCurrentDays = GetDays( systemTime.wDay, systemTime.wMonth, systemTime.wYear %= 100 );

	// 1. Prepare filename
	ConcatStrings( szFileName, m_szStartupPath, c_szForwardsFileName );
	// 2. Open file for reading
	char		szOemFileName[MAX_PATH];
	CharToOem( szFileName, szOemFileName );
	ifstream	in(szOemFileName);
	if (!in.fail()) {
		// 3. consequently read lines
		char	szOemLine[LINE_SIZE];
		TCHAR	szLine[LINE_SIZE];
		while (!in.eof()) {
			in.getline( szOemLine, LINE_SIZE );
			if (in.fail())
				break;
			OemToChar( szOemLine, szLine );
			// 3.1. Get date
			lstrcpyn( fwdInfo.m_szDate, szLine, 7 );
			// 3.1.1. Check date 
			if (g_vars.m_nForwardExpirationTime != 0) {
				WORD	wDay   = (fwdInfo.m_szDate[0] - _T('0'))*10 + (fwdInfo.m_szDate[1] - _T('0'));
				WORD	wMonth = (fwdInfo.m_szDate[2] - _T('0'))*10 + (fwdInfo.m_szDate[3] - _T('0'));
				WORD	wYear  = (fwdInfo.m_szDate[4] - _T('0'))*10 + (fwdInfo.m_szDate[5] - _T('0'));
				int	nDays = GetDays( wDay, wMonth, wYear );
				// skip expired request
				if (nCurrentDays - nDays > g_vars.m_nForwardExpirationTime) {
					m_fForwardsChanged = true;
					continue;
				}
			}
			// 3.2. Get link
			i = 0;
			szFrom = szLine + 7;
			while (*szFrom != _T(' ')) {
				fwdInfo.m_szLink[i] = *szFrom;
				i++;
				szFrom++;
			}
			fwdInfo.m_szLink[i] = _T('\0');
			// 3.3. Get area
			i = 0;
			szFrom++;
			while (*szFrom != _T('\0')) {
				fwdInfo.m_szArea[i] = *szFrom;
				i++;
				szFrom++;
			}
			fwdInfo.m_szArea[i] = _T('\0');
			m_vectForwardInfo.push_back(fwdInfo);
		}
		in.close();
	}

	// in spite of read file result we should set flag
	m_fForwardsRead = true;
}

void CAdrenalin::SaveForwards() {
	TCHAR	szForwardsFileName[MAX_PATH];
	CFile	forwardsFile;

	// 1. Prepare filename
	ConcatStrings( szForwardsFileName, m_szStartupPath, c_szForwardsFileName );
	// 2. Open file for writing
	if (forwardsFile.Open( szForwardsFileName, GENERIC_WRITE, CREATE_ALWAYS )) {
		CForwardInfoVector::iterator	i;
		for (i = m_vectForwardInfo.begin(); i != m_vectForwardInfo.end(); ++i) {
			forwardsFile.WriteStringAsOem( (*i).m_szDate );
			forwardsFile.WriteStringAsOem( _T(" ") );
			forwardsFile.WriteStringAsOem( (*i).m_szLink );
			forwardsFile.WriteStringAsOem( _T(" ") );
			forwardsFile.WriteStringAsOem( (*i).m_szArea );
			forwardsFile.WriteStringAsOem( _T("\r\n") );
		}
		forwardsFile.Close();
	}
	else
		g_vars.m_log << CLog::StdErr(true) 
			     << _T("Error: failed to write forwards file.\n\r")
			     << CLog::StdErr(false);
}

void CAdrenalin::GetErrorMessage(int nErrorCode, LPTSTR szMessage) {
	switch (nErrorCode) {
	case ERROR_UNKNOWN_ARG:
	    lstrcpy( szMessage, _T("Unknown argument") );
	    break;
	case ERROR_HATCH_NO_FILE_DEF:
	    lstrcpy( szMessage, _T("No file defined to hatch") );
	    break;		
	case ERROR_HATCH_NO_AREA_DEF:
	    lstrcpy( szMessage, _T("No area defined to hatch") );
	    break;
	case ERROR_HATCH_EXTRA_ARG:
	    lstrcpy( szMessage, _T("Extra arguments for hatch") );
	    break;
	case ERROR_SYNTAX:
	    lstrcpy( szMessage, _T("Syntax error") );
	    break;
	case ERROR_NUMBER:
	    lstrcpy( szMessage, _T("Wrong number of arguments") );
	    break;
	case ERROR_DUPLICATE:
	    lstrcpy( szMessage, _T("Duplicate usage of a keyword") );
	    break;
	case ERROR_FILE_ABSENT:
	    lstrcpy( szMessage, _T("File absent") );
	    break;
	case ERROR_UNKNOWN_KEYWORD:
	    lstrcpy( szMessage, _T("Unknown keyword") );
	    break;
	case ERROR_FORWARD_DEF:
	    lstrcpy( szMessage, _T("Forward definition") );
	    break;
	case ERROR_ABSENT_DEF:
	    lstrcpy( szMessage, _T("Definition is absent") );
	    break;
	case ERROR_MEMORY:
	    lstrcpy( szMessage, _T("Not enough memory") );
	    break;
	case ERROR_AREA_NOT_FOUND:
	    lstrcpy( szMessage, _T("Area not found") );
	    break;
	case ERROR_MOVE_FAILED:
	    lstrcpy( szMessage, _T("Failed to move file") );
	    break;
	case ERROR_CRC32_FAILED:
	    lstrcpy( szMessage, _T("Failed to calculate CRC32") );
	    break;
	case ERROR_UNKNOWN_AKA:
	    lstrcpy( szMessage, _T("Unknown AKA") );
	    break;
	case ERROR_LNG_READ:
	    lstrcpy( szMessage, _T("Error reading language file") );
	    break;
	case ERROR_LNG_FILE_NOT_FOUND:
	    lstrcpy( szMessage, _T("Language file not found") );
	    break;
	case ERROR_LNG_NUMBER:
	    lstrcpy( szMessage, _T("Wrong number of language messages") );
	    break;
	case ERROR_HATCH_NO_DESC_DEF:
	    lstrcpy( szMessage, _T("Description not defined for hatching") );
	    break;
	case ERROR_HATCH_NO_REPLACES:
	    lstrcpy( szMessage, _T("REPLACES value absent for hatching") );
	    break;
	case ERROR_UNKNOWN_OPT:
	    lstrcpy( szMessage, _T("Unknown option") );
	    break;
	case ERROR_LOW_DISK_SPACE:
	    lstrcpy( szMessage, _T("Low disk space") );
	    break;
	case ERROR_UNDEFINED_ADDRESS:
	    lstrcpy( szMessage, _T("Undefined address for relink") );
	    break;
	case ERROR_WRONG_ADDRESS:
	    lstrcpy( szMessage, _T("Wrong relink address") );
	    break;
	case ERROR_TIMEPURGE_NO_HOURS:
	    lstrcpy( szMessage, _T("Hours for TimePurge not defined") );
	    break;
	case ERROR_TIMEPURGE_INVALID_HOURS:
	    lstrcpy( szMessage, _T("Invalid hours value for TimePurge") );
	    break;
	case ERROR_BASEPURGE_NO_DAYS:
		lstrcpy( szMessage, _T("Days for BasePurge weren't defined") );
		break;
	case ERROR_BASEPURGE_INVALID_DAYS:
		lstrcpy( szMessage, _T("Invalid days value for BasePurge") );
		break;
	case ERROR_HAND_NO_LINK:
		lstrcpy( szMessage, _T("Link isn't specified for Hand command") );
		break;
	case ERROR_HAND_NO_COMMANDS:
		lstrcpy( szMessage, _T("No message commands for Hand command") );
		break;
	default:
		lstrcpy( szMessage, _T("Unknown error") );
	}
}

bool CAdrenalin::AssignFileFlags() {
	CAreasManager*	areasManager = CAreasManager::getInstance();

	int	count = m_vsFileFlags.size();

	for (int i = 0; i < count; i++) {
		PArea	area = areasManager->getArea( m_vsFileAreas[i] );
		if (area == NULL) {
			g_vars.m_log << CLog::StdErr(true) << TEXT("Unknown area for FileFlag: ")
						 << m_vsFileAreas[i]
						 << TEXT("\n");
			m_iErrorCode = ERROR_AREA_NOT_FOUND;
			return (false);
		}
		area->addFileFlag( CFileFlag( m_vsFileFlags[i], m_vsFileMasks[i] ) );
	}

	m_vsFileFlags.clear();
	m_vsFileMasks.clear();
	m_vsFileAreas.clear();

	return (true);
}

void CAdrenalin::TimePurge() {
	// if there in no masks, we shouldn't check links
	bool		fCheckByMask = !m_timePurge.m_vsMasks.empty();
	TCHAR		szAddr[128];
	CBSOManager	bsoManager;
	VTString	vsMsgBodyLines;
	VTString	vsbsoLines;
	vector<bool>	vbbsoLeft;
	TCHAR		szBuf[MAX_PATH];
	char		szOemBuf[MAX_PATH];
	::ifstream	in;
	CTic		tic;
	int		iNewTicsPathLen;
	__int64		i64TresholdTime;
	TCHAR		szFileName[MAX_PATH];
	TCHAR		szShortFileName[MAX_PATH];
	TCHAR		szTicName[MAX_PATH];
	TCHAR		szShortNewTicsPath[MAX_PATH];
	bool		fbsoChanged;			// true, if bso was changed as a result of removing tics and files
	LARGE_INTEGER	li;
	int		i, j;
	TCHAR		szMsgHead[4096];
	TCHAR		szMsgInfo[4096];

	// precalc some vars
	{
		FILETIME	ftCurrent;
		GetSystemTimeAsFileTime( &ftCurrent );
		__int64	i64Hours = ((__int64)m_timePurge.m_iHours)*10000000*60*60;
		li.LowPart  = ftCurrent.dwLowDateTime;
		li.HighPart = ftCurrent.dwHighDateTime;
		i64TresholdTime = li.QuadPart - i64Hours;
	}
	ConvertANSINameToShort( g_vars.m_szNewTicsPath, szShortNewTicsPath );
	iNewTicsPathLen = lstrlen(szShortNewTicsPath);
	MakeDirStructure( g_vars.m_szNetmailPath );
	_stprintf( szMsgHead, m_language[LNG_TIMEPURGE_HEAD], m_timePurge.m_iHours );

	// turn on screen output
	g_vars.m_log << CLog::StdOut(true);

	for (CLinksManager::CIterator	itLink = CLinksManager::getInstance()->begin();
	itLink != CLinksManager::getInstance()->end(); ++itLink) {
		// get next link
		PLink	link = (*itLink);

		// check address by mask if have to
		if (fCheckByMask) {
			link->getAddress().ToString( szAddr );
			VTString::iterator	vit = m_timePurge.m_vsMasks.begin();
			while (vit != m_timePurge.m_vsMasks.end()) {
				if (CheckMask( szAddr, (*vit).c_str() ))
					break;
				vit++;
			}
			// if no appropriate mask was found, make next loop
			if (vit == m_timePurge.m_vsMasks.end())
				continue;
		}

		// log link address
		g_vars.m_log << _T("Link: ") << szAddr << _T("\n");

		// init bso manager
		bsoManager.setAddress( link->getAddress(), link->getFlavourType() );

		// set busy flag
		if (!bsoManager.setBusyFlag()) {
			g_vars.m_log << CLog::StdOut(false) << CLog::StdErr(true)
					 << _T("  failed to create busy flag in BSO\n")
					 << CLog::StdOut(true) << CLog::StdErr(false);
			continue;
		}

		// prepare vectors
		vsMsgBodyLines.clear();
		vsMsgBodyLines.push_back( szMsgHead );
		vsbsoLines.clear();
		vbbsoLeft.clear();

		// read bso file into the memory
		lstrcpy( szBuf, bsoManager.getName() );
		CharToOem( szBuf, szOemBuf );
		in.open( szOemBuf );
		if (in) {
			while (!in.eof()) {
				in.getline( szOemBuf, MAX_PATH );
				if (!in)
					break;
				OemToChar( szOemBuf, szBuf );
				vsbsoLines.push_back( szBuf );
				vbbsoLeft.push_back( true );
			}
			in.close();
		}
		else {
			g_vars.m_log << CLog::StdOut(false) << CLog::StdErr(true)
					 << _T("  failed to open BSO file fo reading:\n")
					 << _T("    ") << szBuf
					 << _T("\n")
					 << CLog::StdOut(true) << CLog::StdErr(false);
		}

		// check presense of lines
		if (!vsbsoLines.empty()) {
			fbsoChanged = false;
			// go throught vsbsoLines
			for (i = 0; i < vsbsoLines.size(); i++) {
				// skip already erased lines
				if (!vbbsoLeft[i])
					continue;
				tstring&	s = vsbsoLines[i];
				// check 2 conditions:
				// 1) tic is in NEWTICS directory
				// 2) it's creation date is less than current-hours
				if (s.length() == 0 || s[0] != _T('^')) {
					continue;
				}
				lstrcpy( szTicName, s.c_str() + 1 );
				if (_tcsncmp( szTicName, 
						  szShortNewTicsPath,
						  iNewTicsPathLen ) == 0 &&
					!HaveSlashes(szTicName + iNewTicsPathLen))
				{
					// first condition is true
					// check second condition
					HANDLE	hTicFile = CreateFile( szTicName, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
					if (hTicFile == INVALID_HANDLE_VALUE) {
						g_vars.m_log << CLog::StdOut(false) << CLog::StdErr(true)
								 << _T("  failed to open TIC file for reading:\n")
								 << _T("    ") << szTicName
								 << _T("\n")
								 << CLog::StdOut(true) << CLog::StdErr(false);
					}
					else {
						// check time
						FILETIME	ftTic;
						if (!GetFileTime( hTicFile, &ftTic, NULL, NULL )) {
							CloseHandle( hTicFile );
							g_vars.m_log << CLog::StdOut(false) << CLog::StdErr(true)
									 << _T("  failed to get filetime of TIC file:\n")
									 << _T("    ") << szTicName
									 << _T("\n")
									 << CLog::StdOut(true) << CLog::StdErr(false);
						}
						else {
							CloseHandle( hTicFile );
							li.LowPart  = ftTic.dwLowDateTime;
							li.HighPart = ftTic.dwHighDateTime;
							__int64	i64TicTime = li.QuadPart;
							if (i64TicTime <= i64TresholdTime) {
								// tic is obsolete
								g_vars.m_log << _T("  TIC file is obsolete:\n    ") << szTicName << _T("\n");
								// read tic
								if (!tic.Read(szTicName)) {
									g_vars.m_log << CLog::StdOut(false) << CLog::StdErr(true)
											 << _T("  failed to read TIC file:\n    ")
											 << szTicName
											 << _T("\n")
											 << CLog::StdOut(true) << CLog::StdErr(false);
								}
								else {
									// we doesn't need this tic anymore
									vbbsoLeft[i] = false;
									fbsoChanged = true;
									if (!DeleteFile( szTicName )) {
										g_vars.m_log << CLog::StdOut(false) << CLog::StdErr(true)
												 << _T("  failed to remove TIC file:\n    ")
												 << szTicName
												 << _T("\n")
												 << CLog::StdOut(true) << CLog::StdErr(false);
									}
									PArea	area = CAreasManager::getInstance()->getArea( tic.m_szArea );
									if (area == NULL) {
										g_vars.m_log << CLog::StdOut(false) << CLog::StdErr(true)
												 << _T("  area ") << tic.m_szArea << _T(" not found\n")
												 << CLog::StdOut(true) << CLog::StdErr(false);
									}
									else {
										// form full file name
										lstrcpy( szFileName, area->getWorkingPath().c_str() );
										if (*tic.m_szFullName != _T('\0'))
											lstrcat( szFileName, tic.m_szFullName );
										else
											lstrcat( szFileName, tic.m_szFile );
										ConvertANSINameToShort( szFileName, szShortFileName );
										// erase line with filename from bso lines
										bool	fErased = false;
										for (j = 0; j < vsbsoLines.size(); j++) {
											if (lstrcmp( vsbsoLines[j].c_str(), szShortFileName ) == 0) {
												vbbsoLeft[j] = false;
												fErased = true;
												break;
											}
										}
										if (!fErased) {
											g_vars.m_log << CLog::StdOut(false) << CLog::StdErr(true)
													 << _T("  file not found in BSO:\n    ") << szShortFileName << _T("\n")
													 << CLog::StdOut(true) << CLog::StdErr(false);

										}
										else {
											g_vars.m_log << _T("  file removed from BSO:\n    ") << szShortFileName << _T("\n");
											_stprintf( szMsgInfo, m_language[LNG_TIMEPURGE_INFO], 
												tic.m_szArea, 
												(tic.m_szFullName[0] == _T('\0')) ? tic.m_szFile : tic.m_szFullName );
											vsMsgBodyLines.push_back( szMsgInfo );
										}
									}
								}									
							}
						}							
					}
				}
			}
			// rewrite bso and create message to the link, if bso has been changed
			if (fbsoChanged) {
				CFile	bsoFile;
				if (!bsoFile.Open( bsoManager.getName(), GENERIC_WRITE, CREATE_ALWAYS )) {
					g_vars.m_log << CLog::StdOut(false) << CLog::StdErr(true)
							 << _T("  failed to open BSO file for writing:\n    ")
							 << bsoManager.getName()
							 << _T("\n")
							 << CLog::StdOut(true) << CLog::StdErr(false);
				}
				else {
					for (i = 0; i < vsbsoLines.size(); i++) {
						if (vbbsoLeft[i]) {
							bsoFile.WriteStringAsOem( vsbsoLines[i].c_str() );
							bsoFile.WriteOemString( "\r\n" );
						}
					}
					bsoFile.Close();
				}
				// create message
				if (vsMsgBodyLines.size() > 1) {
        			CMsg	msg;
					// set receiver (we well reply)
        			msg.m_addrFrom = link->getAddress();
					// set sender
					msg.m_addrTo = link->getOurAddress();
        			strcpy( msg.m_Header.m_szFromUserName, "SysOp" );
        			strcpy( msg.m_Header.m_szToUserName,   "Adrenalin" );
					msg.Reply( ATTR_PRIVATE | ATTR_LOCAL, 0, m_language[LNG_TIMEPURGE_SUBJECT], vsMsgBodyLines );
				}
			}
			else {
				g_vars.m_log << _T("  nothing to purge\n");
			}
		}
		else {
			g_vars.m_log << _T("  nothing to purge\n");
		}

		// clear busy flag
		bsoManager.clearBusyFlag();
	}

	// turn off screen output
	g_vars.m_log << CLog::StdOut(false);
}

const TCHAR*	szFilesDBFileName = _T("files.db");

void CAdrenalin::UpdateFilesDB() {
	CFile	dbFile;
	TCHAR	szFileName[MAX_PATH];

	// prepare full files db file name
	ConcatStrings( szFileName, m_szStartupPath, szFilesDBFileName );

	// open db in append mode
	if (!dbFile.Open( szFileName, GENERIC_WRITE, OPEN_ALWAYS )) {
		g_vars.m_log << CLog::StdErr(true) 
			     << _T("Failed to open '")
			     << szFilesDBFileName
			     << _T("' for appending\n")
			     << CLog::StdErr(false);
		return;
	}
	dbFile.SeekToEnd();

	// write current time
	SYSTEMTIME	stLocal;
	TCHAR		szDateTime[32];
	::GetLocalTime( &stLocal );
	wsprintf( szDateTime, _T("%04hu/%02hu/%02hu %02hu:%02hu:%02hu\n"),
		stLocal.wYear,
		stLocal.wMonth,
		stLocal.wDay,
		stLocal.wHour,
		stLocal.wMinute,
		stLocal.wSecond );

	dbFile.WriteOemString("Date ");
	dbFile.WriteStringAsOem( szDateTime );

	// write new files names from the every area

	int	areasCount = CAreasManager::getInstance()->getAreasCount();
	for (int i = 0; i < areasCount; i++) {
		PArea	area = CAreasManager::getInstance()->getArea( i );
		int	filesCount = area->getNewFilesCount();
		if (filesCount == 0)
			continue;
		dbFile.WriteOemString("Area ");
		dbFile.WriteStringAsOem( area->getName().c_str() );
		dbFile.WriteOemString("\n");
		for (int j = 0; j < filesCount; j++) {
			const tstring&	newFile = area->getNewFile( j );
			if (HasSpaces( newFile.c_str() )) {
				dbFile.WriteOemString("\"");
				dbFile.WriteStringAsOem( newFile.c_str() );
				dbFile.WriteOemString("\"\n");
			}
			else {
				dbFile.WriteStringAsOem( newFile.c_str() );
				dbFile.WriteOemString("\n");
			}
		}
	}
	
	// close db
	dbFile.Close();
}

void CAdrenalin::BasePurge() {
	CCfgParser	parser;
	TCHAR		szFileName[MAX_PATH];
	VTString	vsDBLines;		// we will keep here lines to overwrite db
	bool		fObsoleteFiles;		// flag is set when reading lines of obsolete
						// files
	__int64		i64TresholdTime;
	__int64		i64Date;
	SYSTEMTIME	stDate;
	FILETIME	ftDate;
	LARGE_INTEGER	li;
	TCHAR*		szNameFrom;

	PArea		area = NULL;

	// prepare full files db file name
	ConcatStrings( szFileName, m_szStartupPath, szFilesDBFileName );

	// open files db file
	if (!parser.Open( szFileName )) {
		g_vars.m_log << CLog::StdErr(true)
			     << _T("[!] Failed to open '")
			     << szFilesDBFileName
			     << _T("' for reading\n")
			     << CLog::StdErr(false);
		return;
	}

	// prepare time
	{
		SYSTEMTIME	stCurrent;
		FILETIME	ftCurrent;
		GetLocalTime( &stCurrent );
		SystemTimeToFileTime( &stCurrent, &ftCurrent );
		__int64	i64Days = ((__int64)m_basePurge.m_iDays)*10000000*24*60*60;
		li.LowPart  = ftCurrent.dwLowDateTime;
		li.HighPart = ftCurrent.dwHighDateTime;
		i64TresholdTime = li.QuadPart - i64Days;
	}

	// clear areas
	// TODO: do we need it?
	{
		int	areasCount = CAreasManager::getInstance()->getAreasCount();
		for (int i = 0; i < areasCount; i++) {
			CAreasManager::getInstance()->getArea( i )->forgetNewFiles();
		}
	}

	// consequently read lines from files db file
	tstring	sDate;
	bool	fDateLineStored = false;
	fObsoleteFiles = false;	// we don't know yet what data we are reading
	while (!parser.Eof()) {
		// read line
		parser.ReadLine();
		// check errors
		if (parser.Failed()) {
			/*
			g_vars.m_log << CLog::StdErr(true)
				     << _T("[!] Read error occured while reading '")
				     << szFilesDBFileName
				     << _T("'\n")
				     << CLog::StdErr(false);
			return;
			*/
			break;
		}
		// check for Date line
		if (parser.Count() == 3 && lstrcmpi( parser[0], _T("Date") ) == 0) {
			// keep date line
			sDate           = parser.String();
			fDateLineStored = false;
			// reset flag of obsolete files
			fObsoleteFiles  = false;
			memset( &stDate, 0, sizeof(stDate) );
			// parse date
			if (_stscanf( parser[1], _T("%hu/%hu/%hu"), 
				&stDate.wYear, &stDate.wMonth, &stDate.wDay ) == 3
			    &&
			    _stscanf( parser[2], _T("%hu:%hu:%hu"),
				&stDate.wHour, &stDate.wMinute, &stDate.wSecond ) == 3)
			{
				// convert date
				SystemTimeToFileTime( &stDate, &ftDate );
				li.LowPart  = ftDate.dwLowDateTime;
				li.HighPart = ftDate.dwHighDateTime;
				i64Date     = li.QuadPart;
				if (i64Date <= i64TresholdTime) {
					fObsoleteFiles = true;	// we are now reading obsolete files
					area           = NULL;
				}
			}
			else {
				g_vars.m_log << CLog::StdErr(true)
					     << _T("[!] Warning: failed to parse Date at line ")
					     << parser.GetLine()
					     << _T(" in file'")
					     << szFilesDBFileName
					     << _T("'\n")
					     << CLog::StdErr(false);
			}
			if (!fObsoleteFiles) {
				// date is young, keep line
				vsDBLines.push_back( parser.String() );
			}
		}
		else {
			if (fObsoleteFiles) {
				// skip empty lines
				if (parser.Count() == 0) {
					continue;
				}
				// keep file name
				if (parser.Count() == 1) {
					if (area != NULL) {
						lstrcpy( szFileName, parser[0] );
						if (szFileName[0] == _T('\"'))
							RemoveBoundaryQuotes( szFileName );
						if (szFileName[0] != _T('\0'))
							area->rememberNewFile( szFileName );
					}
					else {
						// keep date
						if (!fDateLineStored) {
							vsDBLines.push_back( sDate );
							fDateLineStored = true;
						}
						// keep line
						vsDBLines.push_back( parser.String() );
					}
				}
				else {
					// check area
					if (parser.Count() == 2 && lstrcmpi( parser[0], _T("Area") ) == 0) {
						area = CAreasManager::getInstance()->getArea( parser[1] );
						if (area != NULL) {
							// check veto
							if (area->hasFlag( CArea::FLAG_VETO_MANUAL_PURGE )) {
								area = NULL;
							}
							else if (!m_basePurge.m_vsAreas.empty()) {
								// check mask
								VTString::iterator	itAreaMask = m_basePurge.m_vsAreas.begin();
								for ( ; itAreaMask != m_basePurge.m_vsAreas.end(); itAreaMask++) {
									if (CheckMask( area->getName().c_str(), (*itAreaMask).c_str() ))
										break;
								}
								if (itAreaMask == m_basePurge.m_vsAreas.end()) {
									// satisfied mask was not found
									area = NULL;
								}
							}
						}
						if (area == NULL) {
							// keep date
							if (!fDateLineStored) {
								vsDBLines.push_back( sDate );
								fDateLineStored = true;
							}
							// keep line
							vsDBLines.push_back( parser.String() );
						}
					}
					else {
						g_vars.m_log << CLog::StdErr(true)
							     << _T("[!] Warning: incorrect number of lexems at line ")
							     << parser.GetLine()
							     << _T(" in file'")
							     << szFilesDBFileName
							     << _T("'\n")
							     << CLog::StdErr(false);
					}
				}
			}
			else {
				// keep line
				vsDBLines.push_back( parser.String() );
			}
		}
	}
	parser.Close();

	// remove obsolete files
	VTString	vsReport;
	VTString	vsRemovedFiles;
	g_vars.m_log << CLog::StdOut(true);

	int	areasCount = CAreasManager::getInstance()->getAreasCount();
	for (int i = 0; i < areasCount; i++) {
		// get area
		area = CAreasManager::getInstance()->getArea( i );
		// check number of files
		int	filesCount = area->getNewFilesCount();
		if (filesCount == 0)
			continue;
		vsReport.push_back( tstring(TEXT("Area: ")) + area->getName() );
		g_vars.m_log << TEXT("Area: ") << area->getName() << _T("\n");
		// prepare file path
		lstrcpy( szFileName, area->getWorkingPath().c_str() );
		szNameFrom = szFileName + lstrlen( szFileName );
		vsRemovedFiles.clear();
		for (int j = 0; j < filesCount; j++) {
			const tstring&	newFile = area->getNewFile( j );
			// prepare full file name
			lstrcpy( szNameFrom, newFile.c_str() );
			// remove file
			if (DeleteFile( szFileName )) {
				vsReport.push_back( tstring(TEXT("         ")) + newFile );
				g_vars.m_log << TEXT("  ") << szNameFrom << TEXT("\n");
				vsRemovedFiles.push_back( newFile );
			}
			else {
				vsReport.push_back( tstring(TEXT("  failed ")) + newFile );
				g_vars.m_log << CLog::StdOut(false)
						 << CLog::StdErr(true)
						 << _T(" [!] Failed to remove '")
						 << szFileName
						 << TEXT("'\n")
						 << CLog::StdErr(false)
						 << CLog::StdOut(true);
			}
		}
		// clear descriptions
		if (!vsRemovedFiles.empty())
			area->clearDescFile( vsRemovedFiles );
	}


	g_vars.m_log << CLog::StdOut(false);
	if (!vsReport.empty() && (g_vars.m_nNoMailNotify == 0)) {
		CMsg	msg;
		// set receiver (we will reply)
		msg.m_addrFrom = g_vars.m_addrMain;
		CharToOem( g_vars.m_szSysOp, msg.m_Header.m_szFromUserName );
		// set sender
		msg.m_addrTo = g_vars.m_addrMain;
		strcpy( msg.m_Header.m_szToUserName, "Adrenalin" );
		msg.Reply( ATTR_PRIVATE | ATTR_LOCAL, 0, _T("List of files removed while purge"), vsReport );
		// create flag crying about new message
		if (g_vars.m_szTossFlagName[0] != _T('\0'))
			CreateFileFlag( g_vars.m_szFixFlagName );
	}


	// overwrite files DB
	CFile	dbFile;
	// prepare full files db file name
	ConcatStrings( szFileName, m_szStartupPath, szFilesDBFileName );
	if (!dbFile.Open( szFileName, GENERIC_WRITE, CREATE_ALWAYS )) {
		g_vars.m_log << CLog::StdErr(true) 
			     << _T("Failed to open '")
			     << szFilesDBFileName
			     << _T("' for writing\n")
			     << CLog::StdErr(false);
		return;
	}
	VTString::iterator	itLine = vsDBLines.begin();
	for ( ; itLine != vsDBLines.end(); itLine++) {
		if ((*itLine).length() > 0)
			dbFile.WriteStringAsOem( (*itLine).c_str() );
		dbFile.WriteOemString("\n");
	}
	dbFile.Close();
}

bool CAdrenalin::CorrectPath(LPTSTR szPath) {
	TCHAR	szTempPath[MAX_PATH];

	// 1. Check quotes
	if (szPath[0] == _T('"'))
            RemoveBoundaryQuotes( szPath );
	// 2. Check empty path
        if (szPath[0] == _T('\0'))
		return (false);

	// keep path in a temporary buffer
	lstrcpy( szTempPath, szPath );

	if (szPath[0] == _T('\\')) {
		// append only startup drive
		ConcatStrings( szPath, m_szStartupDrive, szTempPath );
	} else if (szPath[1] != _T(':')) {
		// append only startup path
		ConcatStrings( szPath, m_szStartupPath, szTempPath );
	}

	return (true);
}


/**
 * Remove double quotes, check empty names and names with path separator
 */
bool CAdrenalin::CorrectName(LPTSTR szName) {
	// 1. Check quotes
	if (szName[0] == TEXT('"'))
            RemoveBoundaryQuotes( szName );
	// 2. Check empty name
        if (szName[0] == TEXT('\0'))
		return false;

	// check path separators
	LPCTSTR	charPointer = szName;
	while (*charPointer != TEXT('\0')) {
		if (*charPointer == TEXT('\\') || *charPointer == TEXT('/')) {
			// name is not a single name :(
			return false;
		}
		charPointer++;
	}
	// name is correct;
	return true;
}


void CAdrenalin::ProcessResendCommand(PLink link, SMsgData& msgData, vector<tstring>& bodyLines)
{
	
	// Proposed steps in this function are:
	// 
	// 1) prepare list of files to resend
	// 2) prepare links to be used as SEEN-BY
	// 3) create tics and fill outbound

	// parse message line tail; it should be in format <area name> [<file mask> ...]
	tstring			areaName;
	vector<tstring>	fileMasks;

	if (!ParseResendCommandTail( msgData.m_sTail, areaName, fileMasks )) {
		// failed to parse command tail
		TCHAR	notifyLine[LINE_SIZE];
		_stprintf( notifyLine, m_language[LNG_RESEND_TAIL_PARSE_ERROR], msgData.m_sTail );
		bodyLines.push_back( notifyLine );
		// TODO: add log line
		return;
	}

	// select area
	PArea	area = CAreasManager::getInstance()->getArea( areaName );
	if (area == NULL)
	{
		// failed to select area
		TCHAR	notifyLine[LINE_SIZE];
		_stprintf( notifyLine, m_language[LNG_AREA_NOT_FOUND], areaName.c_str() );
		bodyLines.push_back( notifyLine );
		// TODO: add log line
		return;
	}

	if (!area->hasFlag( CArea::FLAG_OPEN_FOR_RESEND )) {
		if ((area->getTie( link ) == NULL) || !area->canReceive( link ))
		{
			// user have no access to the specified area
			TCHAR	notifyLine[LINE_SIZE];
			_stprintf( notifyLine, m_language[LNG_AREA_NO_ACCESS], areaName.c_str() );
			bodyLines.push_back( notifyLine );
			// TODO: add log line
			return;
		}
	}

	if (fileMasks.size() == 0) {
		// file masks not specified, let's define one
		fileMasks.push_back( TEXT("*") );
	}

	// get list of files to resend
	vector<tstring>	files;
	vector<tstring> descriptions;
	ReadFilesList( area->getWorkingPath().c_str(), fileMasks, files, descriptions );
	if (files.empty())
	{
		// there is nothing to resend
		// inform user
		TCHAR	notifyLine[LINE_SIZE];
		_stprintf( notifyLine, m_language[LNG_RESEND_NO_FILES], areaName.c_str() );
		bodyLines.push_back( notifyLine );
		return;
	}

	// prepare notification title
	bodyLines.push_back( m_language[LNG_RESEND_FILELIST] );

	// prepare sender address
    SFTNAddress	senderAddress = link->getOurAddress();

	// prepare ticProcessor

	CTic	ticProcessor;
    // set area name
    lstrcpy( ticProcessor.m_szArea, area->getName().c_str() );

	// set Replaces
    ticProcessor.m_szReplaces[0] = TEXT('\0');

    // add all us to seenby list, if have to
    if (g_vars.m_fAllAKASeen) {
	    ticProcessor.m_listSeenby.push_back(g_vars.m_addrMain);
        for (int i = 0; i < g_vars.m_vAKAAddr.size(); i++) {
			ticProcessor.m_listSeenby.push_back(g_vars.m_vAKAAddr[i]);
        }
    }

	// prepare seen-by list
	{
		int	count = area->getTiesCount();
		for (int i = 0; i < count; i++) {
			ConstTiePtr	tie = area->getTie( i );
			// add all linked links; we cannot determine who was in original seen-by list
			if (AddrCmp(link->getAddress(), tie->getLink()->getAddress()) != 0) {
				ticProcessor.m_listSeenby.push_back( tie->getLink()->getAddress() );
			}
		}
	}

	// TODO: don't know for what :) the code below
	vector<PLink>	forwards;
	if (area->getTie( link ) != NULL) {
		forwards.push_back( link );
	}

	ticProcessor.m_addrOrigin = senderAddress;

	for (int i = 0; i < files.size(); i++) {
		LPCTSTR	fullFileName = files[i].c_str();

		// adjust ticProcessor

		// calc crc & size
		ticProcessor.m_iSize = CalcCRC32( fullFileName, ticProcessor.m_uiCRC32 );
		if (ticProcessor.m_iSize == -1) {
			// TODO: process error
		}
		else {

			// check and set file name
			TCHAR	shortFileName[MAX_PATH];
			ConvertANSINameToShort( fullFileName, shortFileName );
			int	index, shortSlashPos, fullSlashPos;
			for (index = 0, shortSlashPos = -1; shortFileName[index] != TEXT('\0'); index++) {
				if (shortFileName[index] == TEXT('\\'))
					shortSlashPos = index;
			}
			shortSlashPos++;
			lstrcpy( ticProcessor.m_szFile, shortFileName + shortSlashPos );
			for (index = 0, fullSlashPos = -1; fullFileName[index] != TEXT('\0'); index++) {
				if (fullFileName[index] == TEXT('\\'))
					fullSlashPos = index;
			}
			fullSlashPos++;
			if (lstrcmpi( fullFileName + fullSlashPos, shortFileName + shortSlashPos ) != 0)
				lstrcpy( ticProcessor.m_szFullName, fullFileName + fullSlashPos );
			else
				ticProcessor.m_szFullName[0] = TEXT('\0');

			// set description
			lstrcpy( ticProcessor.m_szDesc, descriptions[i].c_str() );

			ticProcessor.Forward( link, area, area->getWorkingPath().c_str(), forwards );

			// inform user what file we send
			bodyLines.push_back( tstring(TEXT("    ")) + (fullFileName + fullSlashPos) );
		}
	}
}


bool CAdrenalin::ParseResendCommandTail(const tstring& tail, tstring& areaName, vector<tstring>& fileMasks)
{
	int	i, j;

	// skip leading spaces
	i = 0;
	while ((i < tail.size()) && (tail.at(i) == TEXT(' '))) {
		i++;
	}
	// check for empty tail
	if (i == tail.size()) {
		// there is no data
		return false;
	}

	// get area name
	j = i+1;
	while ((j < tail.size()) && (tail.at(j) != TEXT(' '))) {
		j++;
	}
	areaName = tail.substr( i, j-i );

	// now let's get file masks if they are defined
	fileMasks.clear();

	i = j;
	while (i < tail.size()) {
		// skip spaces
		while ((i < tail.size()) && (tail.at(i) == TEXT(' '))) {
			i++;
		}
		if (i < tail.size()) {
			// suppose we are near next file mask
			TCHAR	stopChar;
			if (tail.at(i) == TEXT('"')) {
				stopChar = TEXT('"');
				i++;
			}
			else {
				stopChar = TEXT(' ');
			}
			// we don't know whether file mask is empty, so let's start from i, not from i+1
			j = i;
			while ((j < tail.size()) && (tail.at(j) != stopChar)) {
				j++;
			}
			if (j == i) {
				// file mask is empty or invalid
				return false;
			}
			tstring	fileMask = tail.substr( i, j-i );
			// skip closing doublequote if it is present
			if ((j < tail.size()) && (tail.at(j) == TEXT('"'))) {
				j++;
			}
			if ((j < tail.size()) && (tail.at(j) != TEXT(' '))) {
				// file mask is incorrect, e.g. tail was something like ' "somearc.rar"somefile1.txt'
				return false;
			}
			i = j;
			fileMasks.push_back( fileMask );
		}
	}
	return true;
}


void CAdrenalin::ReadFilesList(LPCTSTR areaPath, const vector<tstring>& fileMasks, vector<tstring>& files, vector<tstring>& descriptions)
{
	// we need ANSI so let's set it
	BOOL	fileApisWereANSI = ::AreFileApisANSI();
	if (!fileApisWereANSI)
		::SetFileApisToANSI();

    WIN32_FIND_DATA findFileData;
	HANDLE          findFileHandle;
	BOOL            fileFound;

	// prepare file mask to find all files in area directory
	TCHAR           findFileMask[MAX_PATH];
	lstrcpy( findFileMask, areaPath );
	lstrcat( findFileMask, TEXT("*.*") );

	// do search
	findFileHandle = FindFirstFile( findFileMask, &findFileData );
	fileFound      = (findFileHandle != INVALID_HANDLE_VALUE);
	while (fileFound) {
		// skip hidden, temporary files and directories
		if ((findFileData.dwFileAttributes & (FILE_ATTRIBUTE_DIRECTORY | FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_TEMPORARY)) == 0) {
			// skip filelist
			if (_tcscmp( findFileData.cFileName, g_vars.m_szFilelistFileName ) != 0) {
				// check file for masks
				for (int i = 0; i < fileMasks.size(); i++) {
					if (CheckMask( findFileData.cFileName, fileMasks[i].c_str() )) {
						files.push_back( findFileData.cFileName );
						// store empty description
						descriptions.push_back( tstring() );
						break;
					}
				}
			}
		}
		fileFound = FindNextFile( findFileHandle, &findFileData );
	}
	if (findFileHandle != INVALID_HANDLE_VALUE)
		FindClose( findFileHandle );

	// read descriptions

	TCHAR	descFileName[MAX_PATH];
	lstrcpy( descFileName, areaPath );
	lstrcat( descFileName, g_vars.m_szFilelistFileName );

	COemLineReader	lineReader = COemLineReader( descFileName );
	while (lineReader.next()) {

		// get file name from the head of the line

		LPCSTR	line = lineReader.getLine();
		char	stopChar = (line[0] == '"') ? '"' : ' ';		
		int		stopPos  = (stopChar == '"') ? 1 : 0;
		int		len = 0;
		while ((line[stopPos] != '\0') && (line[stopPos] != stopChar) && (len < MAX_PATH)) {
			stopPos++;
			len++;
		}
		if ((line[stopPos] == stopChar) && (len > 0)) {
			// remember file name
			char	oemFileName[MAX_PATH];
			::CopyMemory( oemFileName, (stopChar == '"' ? line+1 : line), len );
			oemFileName[len] = '\0';
			TCHAR	fileName[MAX_PATH];
			::OemToChar( oemFileName, fileName );

			// check file name
			for (int i = 0; i < files.size(); i++) {
				if (descriptions[i].length() == 0) {
					// there is no description yet
					if (lstrcmpi( files[i].c_str(), fileName) == 0) {
						// file name found, let's make description
						if (line[stopPos] == '"') {
							stopPos++;
						}
						while (line[stopPos] == ' ') {
							stopPos++;
						}
						int	startPos = stopPos;
						while (line[stopPos] != '\0') {
							stopPos++;
						}
						if (stopPos > startPos) {
							LPSTR	oemBuffer = new char[stopPos - startPos+1];
							::CopyMemory( oemBuffer, line + startPos, stopPos-startPos );
							oemBuffer[stopPos-startPos] = '\0';
							LPTSTR	buffer = new TCHAR[stopPos-startPos+1];
							::OemToChar( oemBuffer, buffer );
							delete [] oemBuffer;
							descriptions[i] = buffer;
							delete [] buffer;
						}
					}
				}
			}
		}
	}

	// append file path to every file found
	for (int i = 0; i < files.size(); i++) {
		files[i] = tstring(areaPath) + files[i];
	}

	// restore file apis codepage
	if (!fileApisWereANSI)
		::SetFileApisToOEM();

}


/**
 * @return	true if tic was older than max tic age (in hours)
 */
bool CAdrenalin::isTicOld(LPCTSTR ticFileName, int maxTicAge)
{
	HANDLE	ticHandle = CreateFile( ticFileName, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL );
	if (ticHandle == INVALID_HANDLE_VALUE) {
		g_vars.m_log << TEXT("Failed to open TIC file for reading:") << TEXT_EOL 
					 << ticFileName << TEXT_EOL;
		return false;
	}
	// check time
	FILETIME	ticTime;
	// get creation time
	if (!GetFileTime( ticHandle, &ticTime, NULL, NULL )) {
		g_vars.m_log << TEXT("Failed to determine TIC file creation time:") << TEXT_EOL 
					 << ticFileName << TEXT_EOL;
		CloseHandle( ticHandle );
		return false;
	}
	CloseHandle( ticHandle );

	// get system time
	FILETIME	sysTime;
	GetSystemTimeAsFileTime( &sysTime );
	ULARGE_INTEGER	convert;
	convert.LowPart  = sysTime.dwLowDateTime;
	convert.HighPart = sysTime.dwHighDateTime;
	__int64	nanotime = convert.QuadPart;

	convert.LowPart  = ticTime.dwLowDateTime;
	convert.HighPart = ticTime.dwHighDateTime;
	nanotime -= convert.QuadPart;

	__int64	limit = ((__int64)maxTicAge)*10000000*60*60;
	return (nanotime >= limit);
}


int	CAdrenalin::HatchSingleFile(LPCTSTR fileName, LPCTSTR fileDesc) {	
    // check area
	PArea	area = CAreasManager::getInstance()->getArea( m_szHatchArea );
    if (area == NULL) {
	    g_vars.m_log << CLog::StdErr(true)
					 << TEXT("  area ")
					 << m_szHatchArea << TEXT(" doesn't exist") << TEXT_EOL
					 << CLog::StdErr(false);
        return (ERROR_AREA_NOT_FOUND);
    }

    // check file
    if (_taccess( fileName, 0) != 0) {
	    g_vars.m_log << CLog::StdErr(true)
					 << TEXT("  file ") 
					 << fileName << TEXT(" not found") << TEXT_EOL
					 << CLog::StdErr(false);
        return (ERROR_FILE_ABSENT);
    }

    TCHAR       szDestFileName[MAX_PATH];
    TCHAR		szDestFilePath[MAX_PATH];
    TCHAR       szShortFileName[MAX_PATH];
    TCHAR       szFullFileName[MAX_PATH];
    TCHAR       szLine[LINE_SIZE];
    TCHAR       szTimeLine[LINE_SIZE];
    int         i , j, nNameFrom;

    CTic        tic;

    ofstream    out;

    ifstream    in;

    CCfgParser *trafficParser;

    int         nTraffic;
    bool        fDistributedForLinks;

    g_vars.m_log << TEXT("Hatch:") << TEXT_EOL;
    
    // make all fecho directories and set areas traffic to zero
	MakeDirStructure( area->getWorkingPath().c_str() );
	// TODO: should we reset area traffic?
	area->setTraffic( 0 );

    // make tics directories
    MakeDirStructure( g_vars.m_szBadTicsPath );
    MakeDirStructure( g_vars.m_szNewTicsPath );

    // calc crc & size
    tic.m_iSize = CalcCRC32( fileName, tic.m_uiCRC32 );
    if (tic.m_iSize == -1) {
	    g_vars.m_log << CLog::StdErr(true)
					 << TEXT("  failed to calculate CRC32 of ")
					 << fileName << TEXT_EOL
					 << CLog::StdErr(false);
        return (ERROR_CRC32_FAILED);
    }

    // move file into fecho directory if have to
    // first let's find name without path
    for (i = 0, nNameFrom = -1; fileName[i] != ZERO_CHAR; i++) {
        if (fileName[i] == _T('\\'))
            nNameFrom = i;
    }
    nNameFrom++;

    if (m_fLeaveHatchedFile) {
        lstrcpy( szDestFileName, fileName );
		CorrectPath( szDestFileName );
        for (i = 0, j = -1; szDestFileName[i] != TEXT('\0'); i++) {
			if (szDestFileName[i] == PATH_SEPARATOR)
				j = i;
        }
        j++;
        lstrcpy( szDestFilePath, szDestFileName );
        szDestFilePath[j] = ZERO_CHAR;
    }
    else {
		lstrcpy( szDestFilePath, area->getWorkingPath().c_str() );
		lstrcpy( szDestFileName, szDestFilePath );
        lstrcat( szDestFileName, fileName + nNameFrom );
		FileOpResult	foResult = MyCopyFile( fileName, szDestFileName );
		if (foResult != FO_SUCCEEDED) {
			g_vars.m_log << CLog::ResetSingle(new CLog::StdOut())
						 << TEXT("  failed to copy file ")
						 << fileName
						 << TEXT(" into ")
						 << szDestFileName
						 << TEXT_EOL;
			switch (foResult) {
				case FO_FAILED:
					g_vars.m_log << TEXT("  unknown failure reason\n");
					break;
				case FO_SOURCE_UNAVAILABLE:
					g_vars.m_log << TEXT("  source unavailable\n");
					break;
				case FO_DEST_UNAVAILABLE:
					g_vars.m_log << TEXT("  destination unavailable\n");
					break;
				case FO_DEST_INVALID:
					g_vars.m_log << TEXT("  destination file has invalid name\n");
					break;
				case FO_READ_FAILED:
					g_vars.m_log << TEXT("  failed to read source file\n");
					break;
				case FO_WRITE_FAILED:
					g_vars.m_log << TEXT("  failed to write destination file\n");
					break;
			}
			g_vars.m_log << CLog::Pop();
			return (ERROR_MOVE_FAILED);
        }
    }

    g_vars.m_log << TEXT("  file: ") << fileName << TEXT("\n  area: ")
                   << area->getName() << TEXT("\n");
    if (fileDesc[0] != ZERO_CHAR)
        g_vars.m_log << TEXT("  desc: \"") << fileDesc << TEXT("\"\n");
    else
        g_vars.m_log << TEXT("  no description\n");
    if (!IsEmptyStr( m_szHatchReplaces ))
        g_vars.m_log << TEXT("  repl: ") << m_szHatchReplaces << TEXT("\n");
    else
        g_vars.m_log << TEXT("  no Replaces\n");

    if (!m_fLeaveHatchedFile) {
        // update descriptions
        g_vars.m_log << TEXT("  updating description file...");

		filesize_t	fileSize;
		fileSize.LowPart = tic.m_iSize;
        area->updateDescFile( fileName + nNameFrom, fileDesc, fileSize, VTString() );

        g_vars.m_log << TEXT("done") << TEXT_EOL;
    }

    // prepare list of links

	vector<PLink>	forwards;
	
	int	count = area->getTiesCount();
	for (i = 0; i < count; i++) {
		TiePtr	tie  = area->getTie(i);
		PLink	link = tie->getLink();
        // skip passive or unavailable seenby
		if (link->getActivity() != ACTIVITY_USUAL)
            continue;
        // skip links that cannot receive
		if (!area->canReceive( link )) {
			// area disallow to receive files
			continue;
		}
        // add link to list
		forwards.push_back( link );
    }

    SFTNAddress	addr;

    // if only one link is connected to area, set hatch address to its AKA
    if (forwards.size() == 1)
		addr = forwards.back()->getOurAddress();
    else
		addr = g_vars.m_addrMain;

	{
		g_vars.m_log << _T("  updating newfiles.txt file...");
		ConcatStrings( szShortFileName, m_szStartupPath, _T("newfiles.txt") );
		CFile	newFilesFile;
		// open newfiles list
		newFilesFile.Open( szShortFileName, GENERIC_WRITE );
		newFilesFile.SeekToEnd();
		// update newfiles list
		TCHAR	szBuf[1024];
		_stprintf( szBuf, 
				   _T("Area %s\r\nFile %s\r\nSize %ld\r\nDesc %s"),
				   area->getName().c_str(),
				   fileName + nNameFrom,
				   tic.m_iSize,
				   fileDesc );
		lstrcat( szBuf, _T("\r\nOrigin ") );
		addr.ToStringWithoutDomain( szBuf + lstrlen(szBuf) );
		lstrcat( szBuf, _T("\r\nUplink ") );
		addr.ToStringWithoutDomain( szBuf + lstrlen(szBuf) );
		lstrcat( szBuf, _T("\r\n") );
		newFilesFile.WriteStringAsOem( szBuf );
		newFilesFile.Close();
		g_vars.m_log << _T("done\r\n");
	}
    
    // update traffic base
    
    // read areas traffic
    
    ConcatStrings( szFullFileName, m_szStartupPath, TEXT("traffic.dat") );

    *szTimeLine = TEXT('\0');
    trafficParser = new CCfgParser( szFullFileName );
    if (!trafficParser->Failed()) {
        // read date line
        trafficParser->ReadLine();
        if (!trafficParser->Failed())
            lstrcpy( szTimeLine, trafficParser->String() );
        // read data lines
        while (!trafficParser->Eof()) {
            trafficParser->ReadLine();
            if (trafficParser->Failed())
                break;
            if (trafficParser->Count() != 2)
                break;
			PArea	trafficArea = CAreasManager::getInstance()->getArea( (*trafficParser)[0] );
            if (trafficArea == NULL)
                continue;
			if (!StrToInt( (*trafficParser)[1], nTraffic ))
				break;
            trafficArea->setTraffic( nTraffic );
        }
    }
    delete trafficParser;

    area->setTraffic( area->getTraffic() + tic.m_iSize );
    
    {
	    CFile	trafficFile;
	    if (!trafficFile.Open( szFullFileName, GENERIC_WRITE, CREATE_ALWAYS )) {
		    g_vars.m_log << CLog::StdErr(true)
			         << _T("! Error writing traffic base file ")
				 << szFullFileName
				 << _T("\r\n")
				 << CLog::StdErr(false);
	    }
	    else {
			if (IsEmptyStr( szTimeLine )) {
				time_t curTime = time(NULL);
				trafficFile.WriteOemString( ctime( &curTime ) );
			}
			else {
				trafficFile.WriteStringAsOem( szTimeLine );
				trafficFile.WriteOemString( "\r\n" );
			}

			int	areasCount = CAreasManager::getInstance()->getAreasCount();
			for (int i = 0; i < areasCount; i++) {
				PArea	trafficArea = CAreasManager::getInstance()->getArea( i );
				if (trafficArea->getTraffic() > 0) {
					_stprintf( szLine, _T("%s %lu\r\n"), 
						   trafficArea->getName().c_str(), trafficArea->getTraffic() );
					trafficFile.WriteStringAsOem( szLine );
				}
			}
			trafficFile.Close();
	    }
    }

    g_vars.m_log << _T("  creating TICs for connected links...");
    // set area name
    lstrcpy( tic.m_szArea, area->getName().c_str() );
    // check and set file name
    ConvertANSINameToShort( szDestFileName, szShortFileName );
    for (i = 0, j = -1; szShortFileName[i] != _T('\0'); i++) {
        if (szShortFileName[i] == _T('\\'))
            j = i;
    }
    j++;
    lstrcpy( tic.m_szFile, szShortFileName + j );
    if (lstrcmpi( fileName + nNameFrom, szShortFileName + j ) != 0)
        lstrcpy( tic.m_szFullName, fileName + nNameFrom );
    else
	tic.m_szFullName[0] = _T('\0');
    // set desc
    if (!IsEmptyStr( fileDesc ))
        lstrcpy( tic.m_szDesc, fileDesc );
    // set Replaces
    if (!IsEmptyStr( m_szHatchReplaces ))
        lstrcpy( tic.m_szReplaces, m_szHatchReplaces );
    // set Origin to main address, if have to
    if (g_vars.m_fHatchOriginMain || g_vars.m_vAKAAddr.empty())
        tic.m_addrOrigin = g_vars.m_addrMain;
    // add all us to seenby list, if have to
    if (g_vars.m_fAllAKASeen) {
	    tic.m_listSeenby.push_back(g_vars.m_addrMain);
        for (i = 0; i < g_vars.m_vAKAAddr.size(); i++) {
			tic.m_listSeenby.push_back(g_vars.m_vAKAAddr[i]);
        }
    }
    // do tics for all connected links
    // now do tics
    fDistributedForLinks = (!forwards.empty());
    vector<PLink>::iterator	itLinked = forwards.begin();
    while (itLinked != forwards.end()) {
        // set Origin to AKA, if have to
        if (!g_vars.m_vAKAAddr.empty() && !g_vars.m_fHatchOriginMain) {
			tic.m_addrOrigin = (*itLinked)->getOurAddress();
        }
       	tic.Forward( (*itLinked), area, szDestFilePath, forwards );
        ++itLinked;
    }
    // create flag
    if (fDistributedForLinks && !IsEmptyStr( g_vars.m_szHatchFlagName ))
        CreateFileFlag( g_vars.m_szHatchFlagName );
    g_vars.m_log << TEXT("done\n");
    g_vars.m_log << CLog::StdOut(true)
		 << TEXT("File ")
		 << fileName
		 << TEXT(" was successfully hatched into area ")
		 << area->getName()
		 << TEXT("\n")
		 << CLog::StdOut(false);
    return (ERROR_NONE);
}


void CAdrenalin::checkTempBSO() {
	// collect temp entries
	CBSOManager	bsoManager;
	VTString	entries;
	bsoManager.collectTempEntries( entries );

	if (entries.size() > 0) {

		// prepare buffer for file copying
		const	BUFFER_SIZE = 32*1024;
		LPBYTE	buffer = new BYTE[BUFFER_SIZE];

		// prepare vector of BSO route types
		const int	routeTypeCount             = 5;
		FlavourType	routeTypes[routeTypeCount] = 
			{ FT_NORMAL, FT_CRASH, FT_DIRECT, FT_HOLD, FT_IMMEDIATE };

		tstring	tempEntryName;
		tstring	fullEntryName;

		// go through all links
		for (CLinksManager::CIterator itLink = CLinksManager::getInstance()->begin();
			 itLink != CLinksManager::getInstance()->end(); ++itLink)
		{
			PLink	link = (*itLink);
			for (int i = 0; i < routeTypeCount; i++) {
				bsoManager.getTempEntryName( link->getAddress(), routeTypes[i],
											 tempEntryName );
				// go through all entries
				for (VTString::iterator	it = entries.begin(); it != entries.end(); it++) {
					if (lstrcmpi( (*it).c_str(), tempEntryName.c_str() ) == 0) {

						// corresponding entry found

						// setup BSO manager for the link
						bsoManager.setAddress( link->getAddress(), routeTypes[i] );

						if (bsoManager.setBusyFlag()) {
							// prepare full entry name
							bsoManager.getTempEntryPath( fullEntryName );
							fullEntryName += tempEntryName;

							// try to append temp file to the ordinary outbound file

							HANDLE	srcFile = CreateFile( fullEntryName.c_str(),
														  GENERIC_READ, FILE_SHARE_READ,
														  NULL, OPEN_EXISTING,
														  FILE_ATTRIBUTE_NORMAL, NULL );

							if (srcFile != INVALID_HANDLE_VALUE) {

								bool	canBeDeleted = false;

								HANDLE	destFile = CreateFile( bsoManager.getName(), 
												 			   GENERIC_WRITE, 0, NULL, OPEN_ALWAYS,
															   FILE_ATTRIBUTE_NORMAL, NULL);
								if (destFile != INVALID_HANDLE_VALUE) {
									SetFilePointer( destFile, 0, NULL, FILE_END );

									BOOL	goodRead = TRUE, goodWrite = TRUE;
									DWORD	bytesRead, bytesWritten;
									for (;;) {
										goodRead = ReadFile( srcFile, buffer, BUFFER_SIZE, &bytesRead, NULL );
										if (!goodRead || bytesRead <= 0) {
											break;
										}
										goodWrite = WriteFile( destFile, buffer, bytesRead, &bytesWritten, NULL );
										if (!goodWrite || bytesWritten != bytesRead) {
											break;
										}
									}

									CloseHandle( destFile );

									canBeDeleted = (goodRead && goodWrite);
								}

								CloseHandle( srcFile );

								if (canBeDeleted) {
									if (!DeleteFile( fullEntryName.c_str() )) {
										// TODO: log delete failure
									}
								}
							}

							bsoManager.clearBusyFlag();
						}

						break;
					}
				}
			}
		}

		// free buffer
		delete [] buffer;
	}
}


void CAdrenalin::formatLogTime(const SYSTEMTIME& someTime, LPTSTR timeString) {
	_stprintf( timeString, TEXT("%04hu/%02hu/%02hu %02hu:%02hu:%02hu"),
			   someTime.wYear, someTime.wMonth,  someTime.wDay,
			   someTime.wHour, someTime.wMinute, someTime.wSecond );
}


tstring CAdrenalin::getName() {
	return TEXT("Adrenalin v0.70alpha2");
}
